//--------------------------------------------------------
// N2KC MML Compiler for N2KD version 1.0a (beta 2b)
// err_proc.c
// Copyright (C) 2021-2026 Y. Shiokami
// Released under the 3-clause BSD License.
//--------------------------------------------------------

#include "err_proc.h"

#include "../iobuffer.h"
#include "inc_info.h"
#include "src_type.h"
#include "src_list.h"
#include "../set_segm.h"

#include "../util/print.h"
#include "../util/str_proc.h"
#include "../util/fileio.h"


static InputBuffer* in_buf;
static const char ___FAR* addnl_msg; // ǉŕ\񂪂΂ɐݒ.
static const char* op_str = NULL;
static const char* part_str = NULL;
static uint8_t src_idx_set = 0;
static uint16_t src_idx = 0;
static const char* out_filename = NULL; // o̓t@C(G[ɍ폜)

static void in_buf_null_error()
{
	static const char SEGMENT("_TEXT") msg_null_buffer[] = "Error: Input buffer is not set.";
	print_to_stderr_far(msg_null_buffer);
	exit(EXIT_FAILURE);
}

static void print_error(const char ___FAR* err_msg, uint16_t code)
{
	static const char SEGMENT("_TEXT") msg_error_header[] = "Error: ";
	print_to_stderr_far(msg_error_header);
	print_to_stderr_far(err_msg);
	print_to_stderr(" (");
	put_u16_to_stderr(code);
	print_to_stderr(")\n");
	if(addnl_msg != NULL)
	{
		// Error: ƌ킹.
		for(int8_t i = 0; i < 7; ++i) putchar_to_stderr(' ');
		print_to_stderr_far(addnl_msg);
		putchar_to_stderr('\n');
	}
}

static void remove_out_file()
{
	if(out_filename == NULL) return;

	const uint8_t ret = remove_out_file_on_error(out_filename);
	if(!ret)
	{
		// t@C폜s̓bZ[W\̂.
		static const char SEGMENT("_TEXT") msg[] = "Error: Failed to remove output file.";
		print_to_stderr_far(msg);
	}
}

void init_error_info()
{
	in_buf = NULL;
	addnl_msg = NULL;
}

void set_error_info_in_mml(InputBuffer* in_buf_)
{
	in_buf = in_buf_;
}

void set_error_info_addnl_msg(const char ___FAR* msg)
{
	addnl_msg = msg;
}

void set_error_op_str(const char* op_str_)
{
	op_str = op_str_;
}

void set_error_part_str(const char* part_str_)
{
	part_str = part_str_;
}

void set_error_src_idx(const uint16_t idx, const uint8_t unset)
{
	if(unset) src_idx_set = 0;
	else
	{
		src_idx_set = 1;
		src_idx = idx;
	}
}

void set_error_out_filename(const char* out_filename_)
{
	out_filename = out_filename_;
}

void exit_internal_error(const uint16_t memo_code)
{
	putchar_to_stderr('\n'); // ꉞs.
	static const char SEGMENT("_TEXT") msg[] = "Internal error (";
	print_to_stderr_far(msg);
	put_u16_to_stderr(memo_code);
	print_to_stderr(")\n");

	remove_out_file();
	exit(EXIT_FAILURE);
}

void exit_file_op(const FileErrorCode fec, const char* filename)
{
	putchar_to_stderr('\n'); // ꉞs.
	if(filename != NULL)
	{
		static const char SEGMENT("_TEXT") header[] = "File: ";
		print_to_stderr_far(header);
		print_to_stderr(filename);
		putchar_to_stderr('\n');
	}
	print_error(file_error_code_to_str(fec), fec);

	remove_out_file();
	exit(EXIT_FAILURE);
}

void exit_mml_one_back(const MMLErrorCode mec)
{
	if(in_buf == NULL) in_buf_null_error();
	if(in_buf->idx != 0)
		in_buf->idx -= 1;
	exit_mml(mec);
}

void exit_mml_idx(const MMLErrorCode mec, const size_t idx)
{
	in_buf->idx = idx;
	exit_mml(mec);
}

void exit_mml(const MMLErrorCode mec)
{
	if(in_buf == NULL)
	{
		in_buf_null_error();

		remove_out_file();
		exit(EXIT_FAILURE);
	}

	size_t err_idx = in_buf->idx;

	// err_idx ̕s̏ꍇ͉s̑O܂Ŗ߂.
	// ̏ł͉sAĂƖ߂肷邱ƂɂȂ邪,
	// As̐̉sŃG[ɂȂ邱Ƃ͖Ȃ̂ňUCɂȂ.
	while(0 < err_idx && is_newline(in_buf->buf[err_idx])) { --err_idx; }

	// sԍvZ.
	size_t line_counter[N_INCLUDABLE_FILES];
	for(uint8_t i = 0; i < N_INCLUDABLE_FILES; ++i) line_counter[i] = 0;
	for(size_t i = 0; i < in_buf->idx; ++i)
	{
		uint8_t j;
		if(in_buf->buf[i] == '\n')
		{
			// LF.
			j = get_include_info_idx(i);
			line_counter[j] += 1;
		}
		else if(in_buf->buf[i] == '\r')
		{
			if(i < in_buf->read_len - 1 && in_buf->buf[i + 1] == '\n')
			{
				// CRLF.
				++i;
			}
			j = get_include_info_idx(i);
			line_counter[j] += 1;
		}
	}

	// G[̋Nt@C̃t@C𓾂.
	const uint8_t info_idx = get_include_info_idx(err_idx);
	const char* filename = include_info_list[info_idx].filename;

	// G[̋Ns̐擪Ɩ߂.
	size_t line_start = err_idx;
	while(0 < line_start && !is_newline(in_buf->buf[line_start - 1])) { --line_start; }
	size_t line_end = err_idx;
	while(line_end < in_buf->read_len && !is_newline(in_buf->buf[line_end])) { ++line_end; }

	// <t@C>(<sԍ>): part=<part_str>, instruction=<op_str>, src=<src_type> ƕ\.
	static const char SEGMENT("_TEXT") line_close_msg[] = "): ";
	print_to_stderr(filename);
	putchar_to_stderr('(');
	put_st_to_stderr_d(line_counter[info_idx] + 1); // sԍ1IW.
	print_to_stderr_far(line_close_msg);
	uint8_t comma_print = 0;
	if(part_str != NULL)
	{
		comma_print = 1;
		static const char SEGMENT("_TEXT") part_msg[] = "part=";
		print_to_stderr_far(part_msg);
		print_to_stderr(part_str);
	}
	if(op_str != NULL)
	{
		if(comma_print) { putchar_to_stderr(','); putchar_to_stderr(' '); }
		else comma_print = 1;
		static const char SEGMENT("_TEXT") inst_msg[] = "instruction=";
		print_to_stderr_far(inst_msg);
		print_to_stderr(op_str);
	}
	if(src_idx_set)
	{
		if(comma_print) { putchar_to_stderr(','); putchar_to_stderr(' '); }
		static const char SEGMENT("_TEXT") src_msg_0[] = "src=(";
		print_to_stderr_far(src_msg_0);
		put_st_to_stderr_d((size_t)source_list[src_idx].use_number);
		putchar_to_stderr(',');
		print_to_stderr_far(source_type_to_str(source_list[src_idx].source_type));
		putchar_to_stderr(')');
	}
	putchar_to_stderr('\n');

	for(size_t i = line_start; i < line_end; ++i) { putchar_to_stderr(in_buf->buf[i]); }
	putchar_to_stderr('\n');

	if(1 <= err_idx)
	{
		for(size_t i = line_start; i < err_idx; ++i)
			putchar_to_stderr(' ');
	}
	print_to_stderr("^\n");

	exit_mml_err_only(mec);
}

void exit_mml_err_only(const MMLErrorCode mec)
{
	print_error(mml_error_code_to_str(mec), mec);

	remove_out_file();
	exit(EXIT_FAILURE);
}
