//--------------------------------------------------------
// N2KC MML Compiler for N2KD version 1.0a (beta 2b)
// mml.c
// Copyright (C) 2021-2026 Y. Shiokami
// Released under the 3-clause BSD License.
//--------------------------------------------------------

#include <stdint.h>

#include "../util/qsort.h"
#include "../util/string.h"

#include "internal.h"
#include "mml_opl3.h"

#include "opcodes.h"
#include "macro.h"
#include "../util/str_proc.h"
#include "err_proc.h"
#include "../set_segm.h"
#include "consttbl.h"
#include "../util/ar_shift.h"
#include "../util/muldiv32.h"


// MML ̖߂\\.
typedef struct
{
	uint8_t op_strlen;
	char* op_str;
	uint8_t flags;
	void (*func)(InputBuffer*, OutputBuffer*, CompileStatus*);
} PlayOpcode;
// n܂閽߂i[e[ȕ.
typedef struct
{
	const PlayOpcode* opcodes; // n܂ MML ߂̔z(e[u)
	uint8_t n_entries; // e[ũGg.
} OpTableInfo;
// MML ߃RpC֐܂͓n܂閽߂i[e[uւ̃|C^.
// n܂閽߂ꍇ func ɃRpC֐𒼐ڊi[.
typedef union
{
	void (*func)(InputBuffer*, OutputBuffer*, CompileStatus*);
	OpTableInfo table_info;
} OpFuncOrTable;
// MML ߃e[upz̃Gg.
#define PLAY_OP1_FLAG_MULTI (1) // n܂閽߂.
typedef struct
{
	uint8_t flags;
	OpFuncOrTable func_or_table;
} PlayOpcode1;


// ߃RpC̓ꏈ֐.
// c, d, e, f, g, a, b ̃RpC.
static void compile_note(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);
// r ̃RpC
static void compile_rest(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);
// o, <, > ̃RpC.
static void update_octave(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);
// l ̃RpC.
static void update_note_length(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);
// Rg΂.
static void process_comment(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);
// @ ̃RpC.
static void compile_select_voice(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);
// v, V ̃RpC.
static void compile_set_volume(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);
// (, ) ̃RpC
static void compile_change_volume(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);
// [, :, ] ̃RpC.
static void compile_repeat(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);
// y ̃RpC.
static void compile_direct_out(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);
// & ̃RpC.
static void compile_tie(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);
// Q, q ̃RpC.
static void compile_gate_time(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);
// D ̃RpC.
static void compile_detune(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);
// DF ̃RpC.
static void compile_detune(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);
// M ̃RpC.
static void compile_set_lfo_params(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);
// MW ̃RpC.
static void compile_set_lfo_wave(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);
// MS ̃RpC.
static void compile_lfo_switch(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);
// MO ̃RpC.
static void compile_set_lfo_operators(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);
// MH ̃RpC.
static void compile_set_hw_lfo_params(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);
// $ ̃RpC.
static void expand_macro(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);
// ^ ̃RpC.
static void modify_note_length(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);
// p ̃RpC.
static void compile_set_panpot(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);
// | ̃RpC.
static void process_part_filter(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);
// ~ ̃RpC.
static void compile_portament(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);
// L ̃RpC.
static void compile_loop(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);
// T, t ̃RpC.
static void compile_set_tempo(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);
// tn ̃RpC.
static void compile_set_tempo_n(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);
// HS, H ̃RpC.
static void compile_fnum_blk(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);
// @= ̃RpC
static void compile_set_scale(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);

// n܂邩A̖߂̃e[u.
// r̊֌Wŕ񒷂߂OɗĂKv.
//  OpTableInfo \̂ opcodes  far |C^łȂ̂ŁÃZOgɒuƂ͂łȂ.
// @ n܂閽߂̃e[u.
#define N_OPS_START_WITH_AT (2)
static const PlayOpcode play_op_AT_table[N_OPS_START_WITH_AT] = {
	{ 2, "@=", 0, compile_set_scale },
	{ 1, "@", 0, compile_select_voice },
};
// CSM Ή͈U.
// C n܂ MML ߂̃e[u.
//#define N_OPS_START_WITH_C (1)
//static const PlayOpcode play_op_C_table[N_OPS_START_WITH_C] = {
//	{ 3, "CSM", 0, compile_csm },
//};
// D n܂ MML ߂̃e[u.
#define N_OPS_START_WITH_D (2)
static const PlayOpcode play_op_D_table[N_OPS_START_WITH_D] = {
	{ 2, "DF", 0, compile_detune },
	{ 1, "D", 0, compile_detune },
};
// H n܂ MML ߂̃e[u.
#define N_OPS_START_WITH_H (2)
static const PlayOpcode play_op_H_table[N_OPS_START_WITH_H] = {
	{ 2, "HS", 0, compile_fnum_blk },
	{ 1, "H", 0, compile_fnum_blk },
};
// M n܂ MML ߂̃e[u.
#define N_OPS_START_WITH_M (5)
static const PlayOpcode play_op_M_table[N_OPS_START_WITH_M] = {
	{ 2, "MW", 0, compile_set_lfo_wave },
	{ 2, "MS", 0, compile_lfo_switch },
	{ 2, "MO", 0, compile_set_lfo_operators },
	{ 2, "MH", 0, compile_set_hw_lfo_params },
	{ 1, "M",  0, compile_set_lfo_params },
};
// t n܂ MML ߂̃e[u.
#define N_OPS_START_WITH_t (2)
static const PlayOpcode play_op_t_table[N_OPS_START_WITH_t] = {
	{ 2, "tn", 0, compile_set_tempo_n },
	{ 1, "t", 0, compile_set_tempo },
};

// ASCII \(0x20(Xy[X)`0x7e(`_))̃e[u.
// f[^ZOgߖ̂߂ɃR[hZOgɒuĂ̂Œ.
static const PlayOpcode1 SEGMENT("_TEXT") PLAY_OPCODE_1_TABLE[16 * 6 - 1] = {
	{ 0, { .func = NULL } }, // ' '
	{ 0, { .func = NULL } }, // !
	{ 0, { .func = NULL } }, // "
	{ 0, { .func = NULL } }, // #
	{ 0, { .func = expand_macro } }, // $
	{ 0, { .func = NULL } }, // %
	{ 0, { .func = compile_tie } }, // &
	{ 0, { .func = NULL } }, // '
	{ 0, { .func = compile_change_volume } }, // (
	{ 0, { .func = compile_change_volume } }, // )
	{ 0, { .func = NULL } }, // *
	{ 0, { .func = NULL } }, // +
	{ 0, { .func = NULL } }, // ,
	{ 0, { .func = NULL } }, // -
	{ 0, { .func = NULL } }, // .
	{ 0, { .func = NULL } }, // /
	// 0, 1, 2, 3, 4, 5, 6, 7, 8, 9
	{ 0, { .func = NULL } }, { 0, { .func = NULL } }, { 0, { .func = NULL } }, { 0, { .func = NULL } }, { 0, { .func = NULL } }, { 0, { .func = NULL } }, { 0, { .func = NULL } }, { 0, { .func = NULL } }, { 0, { .func = NULL } }, { 0, { .func = NULL } },
	{ 0, { .func = compile_repeat } }, // :
	{ 0, { .func = process_comment } }, // ;
	{ 0, { .func = update_octave } }, // <
	{ 0, { .func = NULL } }, // =
	{ 0, { .func = update_octave } }, // >
	{ 0, { .func = NULL } }, // ?
	{ PLAY_OP1_FLAG_MULTI, { .table_info = { play_op_AT_table, N_OPS_START_WITH_AT } } }, // @
	{ 0, { .func = NULL } }, // A
	{ 0, { .func = NULL } }, // B
	{ 0, { .func = NULL } }, // C // { PLAY_OP1_FLAG_MULTI, { .table_info = { play_op_C_table, N_OPS_START_WITH_C } } }, // C
	{ PLAY_OP1_FLAG_MULTI, { .table_info = { play_op_D_table, N_OPS_START_WITH_D } } }, // D
	{ 0, { .func = NULL } }, // E
	{ 0, { .func = NULL } }, // F
	{ 0, { .func = NULL } }, // G
	{ PLAY_OP1_FLAG_MULTI, { .table_info = { play_op_H_table, N_OPS_START_WITH_H } } }, // H
	{ 0, { .func = NULL } }, // I
	{ 0, { .func = NULL } }, // J
	{ 0, { .func = NULL } }, // K
	{ 0, { .func = compile_loop } }, // L,
	{ PLAY_OP1_FLAG_MULTI, { .table_info = { play_op_M_table, N_OPS_START_WITH_M } } }, // M
	{ 0, { .func = NULL } }, // N
	{ 0, { .func = NULL } }, // O
	{ 0, { .func = NULL } }, // P
	{ 0, { .func = compile_gate_time } }, // Q
	{ 0, { .func = NULL } }, // R
	{ 0, { .func = NULL } }, // S
	{ 0, { .func = compile_set_tempo } }, // T
	{ 0, { .func = NULL } }, // U
	{ 0, { .func = compile_set_volume } }, //V
	{ 0, { .func = NULL } }, // W
	{ 0, { .func = NULL } }, // X
	{ 0, { .func = NULL } }, // Y
	{ 0, { .func = NULL } }, // Z
	{ 0, { .func = compile_repeat } }, // [
	{ 0, { .func = NULL } }, // "\\"
	{ 0, { .func = compile_repeat } }, // ]
	{ 0, { .func = modify_note_length } }, // ^
	{ 0, { .func = NULL } }, // _
	{ 0, { .func = NULL } }, // `
	{ 0, { .func = compile_note } }, // a
	{ 0, { .func = compile_note } }, // b
	{ 0, { .func = compile_note } }, // c
	{ 0, { .func = compile_note } }, // d
	{ 0, { .func = compile_note } }, // e
	{ 0, { .func = compile_note } }, // f
	{ 0, { .func = compile_note } }, // g
	{ 0, { .func = NULL } }, // h
	{ 0, { .func = NULL } }, // i
	{ 0, { .func = NULL } }, // j
	{ 0, { .func = NULL } }, // k
	{ 0, { .func = update_note_length } }, // l
	{ 0, { .func = NULL } }, // n
	{ 0, { .func = NULL } }, // m
	{ 0, { .func = update_octave } }, // o
	{ 0, { .func = compile_set_panpot } }, // p
	{ 0, { .func = compile_gate_time } }, // q
	{ 0, { .func = compile_rest } }, // r
	{ 0, { .func = NULL } }, // s
	{ PLAY_OP1_FLAG_MULTI, { .table_info = { play_op_t_table, N_OPS_START_WITH_t } } }, // t
	{ 0, { .func = NULL } }, // u
	{ 0, { .func = compile_set_volume } }, // v
	{ 0, { .func = NULL } }, // w
	{ 0, { .func = NULL } }, // x
	{ 0, { .func = compile_direct_out } }, // y
	{ 0, { .func = NULL } }, // z
	{ 0, { .func = NULL } }, // {
	{ 0, { .func = process_part_filter } }, // |
	{ 0, { .func = NULL } }, // }
	{ 0, { .func = compile_portament } }, // ~
};

// farstack ŏĂ̂ŁAɃ|C^֐Ăяoɂ͂n.
static uint8_t tmp_u8_static;
static int16_t tmp_i16_static;
static uint32_t tmp_u32_static;

// compile_mml_one() eRpC֐ĂяoOɐݒ肷RpCΏۂ̖߂̊Jnʒu.
static size_t op_start_idx = 0;

// NbNۑobt@ւ̉Zs.
static void add_clock_buf(CompileStatus* cs, const uint16_t clk);
static void sub_clock_buf(CompileStatus* cs, const uint16_t clk);

// |^gƃ^CEX[ŃIN^[u̕ύX󂯕t邽߂̊֐.
static uint8_t proc_consecutive_octave_update(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs);

// e|ύXߏo͋ʊ֐.
static void output_set_tempo(OutputBuffer* out_buf, uint8_t timer_b_val);
// timer_b C^[o^C}(8253A)ɏlɕϊ֐.
static uint16_t convert_timer_b_to_8253a(const uint8_t timer_b, const uint8_t sysclk);

// op_start_idx ̈ʒu MML G[ɂ.
// op_start_idx LȊeRpC֐̌Ăяoz肵Ă.
void exit_mml_op_start_idx(MMLErrorCode mec)
{
	exit_mml_idx(mec, op_start_idx);
}

// n܂閽߂ꍇ̃RpC.
// PLAY_OPCODE_1_TABLE f[^ZOgɂȂ̂ table  far |C^Ŏ󂯎.
static void compile_mml_one_multi(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs, const OpTableInfo ___FAR* table)
{
	for(uint8_t i = 0; i < table->n_entries; ++i)
	{
		const PlayOpcode* op = table->opcodes + i;

		// obt@̎c蕶񒷃`FbN.
		const size_t remain_len = in_buf->read_len - in_buf->idx;
		if(remain_len < op->op_strlen) continue;

		if(strncmp(in_buf->buf + in_buf->idx, op->op_str, op->op_strlen) == 0)
		{
			set_error_op_str(op->op_str);
			op_start_idx = in_buf->idx;
			(*(op->func))(in_buf, out_buf, cs);
			set_error_op_str(NULL);

			skip_blank_exit_macro(in_buf, cs); // ߒ̋󔒂΂.
			return;
		}
	}

	// RpCł閽߂̕񂪌Ȃ.
	set_error_op_str(NULL);
	exit_mml(MMLEC_UNKNOWN_OP_CH);
}

// in_buf  idx ̈ʒuǂ MML 1RpC(Oɋ󔒂XLbv肵Ȃ)
void compile_mml_one(InputBuffer* in_buf, OutputBuffer *out_buf, CompileStatus* cs)
{
	// MML ̖ߕgăe[u.
	const uint8_t ch = in_buf->buf[in_buf->idx];
	if(ch < 0x20 || 0x7e < ch)
	{
		// ͈͊O.
		// TODO: 䕶̂܂ܕ\̂͂܂悭Ȃ̂ŉP.
		exit_mml(MMLEC_UNKNOWN_OP_CH);
	}

	// ___FAR łȂ SEGMENT("_TEXT") gȂƃZOg񂪂Ȃ.
	const PlayOpcode1 SEGMENT("_TEXT")* pop1 = PLAY_OPCODE_1_TABLE + (ch - 0x20);
	if(pop1->flags & PLAY_OP1_FLAG_MULTI)
	{
		// ₪ꍇ.
		compile_mml_one_multi(in_buf, out_buf, cs, &(pop1->func_or_table.table_info));
	}
	else
	{
		// RpC֐.
		if(pop1->func_or_table.func == NULL) exit_mml(MMLEC_UNKNOWN_OP_CH);

		static char op_str_buf[2] = { '\0', '\0' };
		op_str_buf[0] = ch;
		set_error_op_str(op_str_buf);
		op_start_idx = in_buf->idx;
		(*(pop1->func_or_table.func))(in_buf, out_buf, cs);
		set_error_op_str(NULL);

		skip_blank_exit_macro(in_buf, cs); // ߒ̋󔒂΂.
	}
}

void compile_note(InputBuffer* in_buf, OutputBuffer *out_buf, CompileStatus* cs)
{
	check_outbuf_remain_size(2, out_buf->wrote_len);

	// Ŏgԍɕϊ.
	get_note_num(in_buf, &tmp_u8_static, cs);
	const uint8_t note_num = tmp_u8_static;

	// 擾.
	get_note_length(in_buf, &tmp_u8_static, cs); // get_note_num()  skip_blank_exit_macro() ςŏoĂ.

	// ߏo.
	cs->part_status.prev_note_pos = out_buf->wrote_len;

	out_buf->buf[out_buf->wrote_len + 0] = note_num_to_keyon_op(note_num);
	out_buf->buf[out_buf->wrote_len + 1] = tmp_u8_static - 1; // note_len
	out_buf->wrote_len += 2;

	cs->part_status.flag_common_0 |= PFLAG_COMM_0_PORT_OK;
	cs->part_status.flag_common_0 &= ~(PFLAG_COMM_0_INF_LOOP);
	add_clock_buf(cs, tmp_u8_static);
}

void compile_rest(InputBuffer* in_buf, OutputBuffer *out_buf, CompileStatus* cs)
{
	check_outbuf_remain_size(2, out_buf->wrote_len);
	in_buf->idx += 1;

	skip_blank_exit_macro(in_buf, cs); // EOF ĂG[Ȃ(ftHgg).
	get_note_length(in_buf, &tmp_u8_static, cs);

	// ߏo.
	cs->part_status.prev_note_pos = out_buf->wrote_len;

	out_buf->buf[out_buf->wrote_len + 0] = OP_REST;
	out_buf->buf[out_buf->wrote_len + 1] = tmp_u8_static - 1; // note_len
	out_buf->wrote_len += 2;

	cs->part_status.flag_common_0 &= ~(PFLAG_COMM_0_INF_LOOP);
	add_clock_buf(cs, tmp_u8_static);
}

void update_octave(InputBuffer* in_buf, OutputBuffer *out_buf, CompileStatus* cs)
{
	(void)out_buf; //unused

	const char ch = in_buf->buf[(in_buf->idx)++];
	if(ch == 'o')
	{
		skip_blank_check_eof_exit_macro(in_buf, cs);

		StrProcErrorCode sec = str_to_num_u8(in_buf, &tmp_u8_static, OCTAVE_MIN, OCTAVE_MAX);
		if(sec != SEC_NO_ERROR) exit_str_proc(sec);
		else cs->part_status.octave = tmp_u8_static; // new_octave
	}
	else // if((ch == '<') || (ch == '>'))
	{
		uint8_t down = (ch == '<');
		if(cs->oct_swap) down = !down;
		if(down)
		{
			if(cs->part_status.octave <= OCTAVE_MIN) exit_mml_one_back(MMLEC_OCTAVE_OUT_OF_RANGE);
			else cs->part_status.octave -= 1;
		}
		else
		{
			if(OCTAVE_MAX <= cs->part_status.octave) exit_mml_one_back(MMLEC_OCTAVE_OUT_OF_RANGE);
			else cs->part_status.octave += 1;
		}
	}
}

void compile_select_voice(InputBuffer* in_buf, OutputBuffer *out_buf, CompileStatus* cs)
{
	check_outbuf_remain_size(2, out_buf->wrote_len);

	in_buf->idx += 1;
	skip_blank_check_eof_exit_macro(in_buf, cs);

	StrProcErrorCode sec = str_to_num_u8(in_buf, &tmp_u8_static, 0x00, 0xff);
	if(sec != SEC_NO_ERROR) exit_str_proc(sec);

	const uint8_t at_num = tmp_u8_static;
	if(cs->voice_registered[at_num] == 0)
	{
		extern const char ___FAR msg_sel_voice_undef[];
		set_error_info_addnl_msg(msg_sel_voice_undef);
		exit_mml_op_start_idx(MMLEC_INVALID_VOICE_IDX);
	}

	// ŗL.
	if(cs->header->source_type == SRC_TYPE_OPL3)
	{
		const uint8_t is_4op = check_voice_is_4op(cs->voice_buf, cs->voice_idx_table[at_num]);
		if(!is_4op) cs->part_status.flag_specific_0 &= ~(PFLAG_OPL3_0_4OP);
		else
		{
			cs->part_status.flag_specific_0 |= PFLAG_OPL3_0_4OP;
			// 4op ߂gȂp[gŎgĂG[ɂ.
			const char pch = cs->part_char;
			if(pch != 'A' && pch != 'B' && pch != 'C' && pch != 'J' && pch != 'K' && pch != 'L')
			{
				extern const char ___FAR msg_sel_voice_4op[];
				set_error_info_addnl_msg(msg_sel_voice_4op);
				exit_mml_op_start_idx(MMLEC_INVALID_VOICE_IDX);
			}
		}
	}

	// ߏo.
	out_buf->buf[out_buf->wrote_len + 0] = OP_SELECT_VOICE;
	out_buf->buf[out_buf->wrote_len + 1] = cs->voice_idx_table[at_num];
	out_buf->wrote_len += 2;
}

void compile_set_volume(InputBuffer* in_buf, OutputBuffer *out_buf, CompileStatus* cs)
{
	if(cs->header->source_type != SRC_TYPE_OPL3) exit_mml(MMLEC_UNSUPPORTED_SOURCE); //  OPL3 ̂ݑΉ.
	check_outbuf_remain_size(2, out_buf->wrote_len);

	const char ch = in_buf->buf[(in_buf->idx)++];

	skip_blank_check_eof_exit_macro(in_buf, cs);

	const uint8_t lower_v = (ch == 'v');

	const uint8_t v_max = lower_v ? (OPL3_VCONV_TABLE_SIZE - 1) : 63;
	StrProcErrorCode sec = str_to_num_u8(in_buf, &tmp_u8_static, 0x00, v_max);
	if(sec != SEC_NO_ERROR) exit_str_proc(sec);

	uint8_t v_num = tmp_u8_static;
	if(lower_v) v_num = OPL3_VCONV_TABLE[v_num];

	// ߏo.
	out_buf->buf[out_buf->wrote_len + 0] = OP_SET_VOLUME;
	out_buf->buf[out_buf->wrote_len + 1] = v_num;
	out_buf->wrote_len += 2;
}

void compile_change_volume(InputBuffer* in_buf, OutputBuffer *out_buf, CompileStatus* cs)
{
	if(cs->header->source_type != SRC_TYPE_OPL3) exit_mml(MMLEC_UNSUPPORTED_SOURCE); //  OPL3 ̂ݑΉ.
	check_outbuf_remain_size(2, out_buf->wrote_len);

	const char ch = in_buf->buf[in_buf->idx];
	in_buf->idx += 1;
	skip_blank_exit_macro(in_buf, cs);

	uint8_t fine_flag = 0;
	if(in_buf->idx < in_buf->read_len && in_buf->buf[in_buf->idx] == '%')
	{
		fine_flag = 1;
		in_buf->idx += 1;
		skip_blank_exit_macro(in_buf, cs);
	}

	// Gcȏꍇ(OPL3ł)2{ω.
	const uint8_t max = fine_flag ? 127 : 127 / 2; // ωʂ͕t.
	StrProcErrorCode sec = str_to_num_u8(in_buf, &tmp_u8_static, 0, max);
	if(sec != SEC_NO_ERROR)
	{
		if(sec == SEC_NOT_NUM) tmp_u8_static = 1; // lȂωʂ1
		else exit_str_proc(sec);
	}
	if(tmp_u8_static == 0) return;

	int8_t vol_d = (int8_t)tmp_u8_static;
	if(!fine_flag) vol_d *= 2;
	if(ch == '(') vol_d = -vol_d;

	// ߏo.
	out_buf->buf[out_buf->wrote_len + 0] = OP_CHANGE_VOLUME;
	out_buf->buf[out_buf->wrote_len + 1] = (uint8_t)vol_d;
	out_buf->wrote_len += 2;
}

void compile_repeat(InputBuffer* in_buf, OutputBuffer *out_buf, CompileStatus* cs)
{
	const char ch = in_buf->buf[(in_buf->idx)++];

	if(ch == '[')
	{
		check_outbuf_remain_size(4, out_buf->wrote_len);
		if(REPEAT_NEST_MAX <= cs->part_status.n_repeat_nest) exit_mml_one_back(MMLEC_MULTI_REPEAT_ESC);

		out_buf->buf[out_buf->wrote_len + 0] = OP_REPEAT_START;
		out_buf->wrote_len += 4;

		cs->part_status.repeat_start_pos[cs->part_status.n_repeat_nest] = out_buf->wrote_len - 4; // s[gJn߂̈ʒuL.
		cs->part_status.n_repeat_nest += 1;
	}
	else if(ch == ':')
	{
		check_outbuf_remain_size(1, out_buf->wrote_len);
		if(cs->part_status.n_repeat_nest == 0) exit_mml_one_back(MMLEC_NO_REPEAT_START);
		const uint8_t mask = (1 << (cs->part_status.n_repeat_nest - 1));
		if(cs->part_status.repeat_esc_flag & mask) exit_mml_one_back(MMLEC_MULTI_REPEAT_ESC);
		cs->part_status.repeat_esc_flag |= mask;

		// ߏo.
		out_buf->buf[out_buf->wrote_len] = OP_REPEAT_ESCAPE;
		out_buf->wrote_len += 1;
	}
	else if(ch == ']')
	{
		const uint8_t clk_idx = cs->part_status.n_repeat_nest;

		check_outbuf_remain_size(1, out_buf->wrote_len);
		if(cs->part_status.n_repeat_nest == 0) exit_mml_one_back(MMLEC_NO_REPEAT_START);
		cs->part_status.n_repeat_nest -= 1;
		cs->part_status.repeat_esc_flag &= ~(1 << cs->part_status.n_repeat_nest);

		skip_blank_check_eof_exit_macro(in_buf, cs);

		// s[g񐔎擾.
		StrProcErrorCode sec = str_to_num_u8(in_buf, &tmp_u8_static, 0x02, 0xff);
		if(sec == SEC_NOT_NUM) tmp_u8_static = 2;
		else if(sec != SEC_NO_ERROR) exit_str_proc(sec);

		// ߏo.
		out_buf->buf[out_buf->wrote_len + 0] = OP_REPEAT_END;
		out_buf->wrote_len += 1;

		// Ή郊s[gJn߂̌Ƀs[g񐔁As[g͈͂̒(gGfBA)L^.
		const uint16_t idx = cs->part_status.repeat_start_pos[cs->part_status.n_repeat_nest];
		out_buf->buf[idx + 1] = tmp_u8_static; // n_repeat
		const uint16_t rep_range_len = out_buf->wrote_len - idx;
		*(uint16_t ___FAR*)(out_buf->buf + idx + 2) = rep_range_len;

		// NbNvZ.
		const uint16_t clk_esc = cs->part_status.clock_buf_esc[clk_idx];
		uint16_t clk = cs->part_status.clock_buf[clk_idx] + clk_esc; // JԂԑŜ1񕪂̃NbN.
		clk *= (uint16_t)tmp_u8_static;
		clk -= clk_esc;
		add_clock_buf(cs, clk);
		cs->part_status.clock_buf[clk_idx] = cs->part_status.clock_buf_esc[clk_idx] = 0;
	}
}

void compile_direct_out(InputBuffer* in_buf, OutputBuffer *out_buf, CompileStatus* cs)
{
	check_outbuf_remain_size(3, out_buf->wrote_len);

	in_buf->idx += 1;
	skip_blank_check_eof_exit_macro(in_buf, cs);

	// AhX擾.
	StrProcErrorCode sec = str_to_num_u8(in_buf, &tmp_u8_static, 0x00, 0xff);
	if(sec != SEC_NO_ERROR) exit_str_proc(sec);
	const uint8_t addr_num = tmp_u8_static;

	// J}邩`FbN.
	skip_one_comma(in_buf, cs, 1);

	// ol擾.
	sec = str_to_num_u8(in_buf, &tmp_u8_static, 0x00, 0xff);
	if(sec != SEC_NO_ERROR) exit_str_proc(sec);

	// ߏo.
	out_buf->buf[out_buf->wrote_len + 0] = OP_DIRECT_OUT;
	out_buf->buf[out_buf->wrote_len + 1] = addr_num;
	out_buf->buf[out_buf->wrote_len + 2] = tmp_u8_static; // val_num
	out_buf->wrote_len += 3;
}

void compile_tie(InputBuffer* in_buf, OutputBuffer *out_buf, CompileStatus* cs)
{
	if(cs->part_status.prev_note_pos == 0)
	{
		extern const char ___FAR msg_no_prev_note[];
		set_error_info_addnl_msg(msg_no_prev_note);
		exit_mml(MMLEC_INVALID_TIE);
	}

	// ^C̖߂̏o͂̂ݐɍς܂.
	// 1oCgo͂]TKv.
	check_outbuf_remain_size(1, out_buf->wrote_len);
	// O̔߂̌1oCg󂯂.
	uint16_t idx = out_buf->wrote_len - 1;
	for(; cs->part_status.prev_note_pos + 2 <= idx; --idx) { out_buf->buf[idx + 1] = out_buf->buf[idx]; }
	// O̔ߒɃ^C߂o.
	out_buf->buf[idx + 1] = OP_TIE;
	out_buf->wrote_len += 1;

	in_buf->idx += 1;
	skip_blank_check_eof_exit_macro(in_buf, cs);

	// xqƂĂȂ`FbN.
	if(in_buf->buf[in_buf->idx] == 'r' || out_buf->buf[cs->part_status.prev_note_pos] == OP_REST)
	{
		extern const char ___FAR msg_tie_rest[];
		set_error_info_addnl_msg(msg_tie_rest);
		exit_mml(MMLEC_INVALID_TIE);
	}

	const uint8_t has_oct = proc_consecutive_octave_update(in_buf, out_buf, cs);
	if(is_pitch_name_char(in_buf->buf[in_buf->idx]))
	{
		// & ̌ c,d,e,f,g,a,b ΁Acompile_note() Ăяo.
		compile_note(in_buf, out_buf, cs);
	}
	else
	{
		// łȂΉ擾(IN^[uw肪ΉKvȂ̂ŃG[)
		if(has_oct) exit_mml(MMLEC_NOT_NOTE_OP_CH);
		MMLErrorCode mec = get_note_length(in_buf, &tmp_u8_static, cs);
		if(mec == MMLEC_NOT_NUM) exit_mml(mec);

		// O̔߂Ɠ(Ⴕ͋x)Ǝ擾o.
		// 2oCgo͂]TKv.
		check_outbuf_remain_size(2, out_buf->wrote_len);
		out_buf->buf[out_buf->wrote_len + 0] = out_buf->buf[cs->part_status.prev_note_pos];
		out_buf->buf[out_buf->wrote_len + 1] = tmp_u8_static - 1; // note_length
		cs->part_status.prev_note_pos = out_buf->wrote_len;
		out_buf->wrote_len += 2;
		add_clock_buf(cs, tmp_u8_static);
	}
}

void compile_gate_time(InputBuffer* in_buf, OutputBuffer *out_buf, CompileStatus* cs)
{
	const char ch = in_buf->buf[(in_buf->idx)++]; // 'Q'  'q'

	check_outbuf_remain_size(2, out_buf->wrote_len);
	skip_blank_check_eof_exit_macro(in_buf, cs);

	const uint8_t lower_q = (ch == 'q');
	uint8_t q_min, q_max, q_op;
	if(lower_q) { q_min = 0; q_max = 255; q_op = OP_GATE_q; }
	else { q_min = 0; q_max = cs->q_reso; q_op = OP_GATE_Q; }

	StrProcErrorCode sec = str_to_num_u8(in_buf, &tmp_u8_static, q_min, q_max);
	if(sec != SEC_NO_ERROR) exit_str_proc(sec);

	// lower_q ł (q_max == 255) Ȃ̂ lower_q ̃`FbN͂ĂȂ.
	if(q_max == 8) tmp_u8_static *= 2; // ̕\ɕϊ.

	// ߏo.
	out_buf->buf[out_buf->wrote_len + 0] = q_op;
	out_buf->buf[out_buf->wrote_len + 1] = tmp_u8_static;
	out_buf->wrote_len += 2;
}

void compile_detune(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs)
{
	check_outbuf_remain_size(3, out_buf->wrote_len);

	// D ߂ DF ߂.
	uint8_t fine_detune = 0;
	if(((in_buf->idx + 1) < in_buf->read_len) && in_buf->buf[in_buf->idx + 1] == 'F')
		fine_detune = 1;

	in_buf->idx += (fine_detune ? 2 : 1);
	skip_blank_check_eof_exit_macro(in_buf, cs);

	StrProcErrorCode sec = str_to_num_i16(in_buf, &tmp_i16_static, INT16_MIN, INT16_MAX);
	if(sec != SEC_NO_ERROR) exit_str_proc(sec);

	// ߏo.
	out_buf->buf[out_buf->wrote_len + 0] = fine_detune ? OP_FINE_DETUNE : OP_DETUNE;
	*((int16_t ___FAR*)(out_buf->buf + out_buf->wrote_len + 1)) = tmp_i16_static;
	out_buf->wrote_len += 3;
}

void compile_set_lfo_params(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs)
{
	MMLErrorCode mec;
	StrProcErrorCode sec;

	in_buf->idx += 1;
	check_eof_exit_macro(in_buf, cs);

	// ǂ LFO g.
	const int8_t channel = get_lfo_channel(in_buf->buf[(in_buf->idx)++]);
	if(channel < 0) exit_mml_one_back(MMLEC_UNEXPECTED_CHAR);
	skip_blank_check_eof_exit_macro(in_buf, cs);

	uint8_t delay;
	if(in_buf->buf[in_buf->idx] == 'l')
	{
		// ̋L@gꍇ.
		in_buf->idx += 1;
		skip_blank_check_eof_exit_macro(in_buf, cs);
		mec = get_note_length(in_buf, &tmp_u8_static, cs);
		if(mec == MMLEC_NOT_NUM) exit_mml(mec);
	}
	else
	{
		// NbNw.
		sec = str_to_num_u8(in_buf, &tmp_u8_static, 0x00, 0xff);
		if(sec != SEC_NO_ERROR) exit_str_proc(sec);
	}
	delay = tmp_u8_static;

	// speed, step, count͑Ŝ܂܂̒l̂ for Ńp[X.
	uint8_t speed_step_count[3];
	for(uint8_t i = 0; i <= 2; ++i)
	{
		// ȍ~̓J}؂.
		skip_one_comma(in_buf, cs, 1);

		// step  int8_t ŁAȊO uint8_t Ȃ̂Œ.
		if(i != 1)
		{
			sec = str_to_num_u8(in_buf, &tmp_u8_static, 0x01, 0xff); // speed, count ̍ŏl1
			if(sec != SEC_NO_ERROR) exit_str_proc(sec);
			speed_step_count[i] = tmp_u8_static;
		}
		else
		{
			sec = str_to_num_i16(in_buf, &tmp_i16_static, -127, 127); // neg Ȃ悤ɂ.
			if(sec != SEC_NO_ERROR) exit_str_proc(sec);
			speed_step_count[i] = ((uint8_t*)(&tmp_i16_static))[0]; // uint8_t ƂďE.
		}
	}

	// wave()
	uint8_t wave = 0;
	uint8_t has_wave = skip_one_comma(in_buf, cs, 0);
	if(has_wave) // J}LŖ߂Ă wave ̃p[X݂.
	{
		check_eof_exit_macro(in_buf, cs);
		sec = str_to_num_u8(in_buf, &tmp_u8_static, 0x00, N_LFO_WAVE_TYPE - 1);
		if(sec != SEC_NO_ERROR) exit_str_proc(sec);
		wave = tmp_u8_static;
	}

	// wave  g`I𖽗߂ɕϊčŏɏo͂. 3oCgo͂]TKv.
	if(has_wave)
	{
		check_outbuf_remain_size(3, out_buf->wrote_len);
		out_buf->buf[out_buf->wrote_len + 0] = OP_LFO_SET_WAVE;
		out_buf->buf[out_buf->wrote_len + 1] = (uint8_t)channel;
		out_buf->buf[out_buf->wrote_len + 2] = wave;
		out_buf->wrote_len += 3;
	}
	// LFO ݒo. 6oCg̗]TKv.
	check_outbuf_remain_size(6, out_buf->wrote_len);
	out_buf->buf[out_buf->wrote_len + 0] = OP_LFO_SET_PARAM;
	out_buf->buf[out_buf->wrote_len + 1] = (uint8_t)channel;
	out_buf->buf[out_buf->wrote_len + 2] = delay;
	out_buf->buf[out_buf->wrote_len + 3] = speed_step_count[2] - 1; // count(n_step)
	out_buf->buf[out_buf->wrote_len + 4] = speed_step_count[0] - 1; // speed
	out_buf->buf[out_buf->wrote_len + 5] = speed_step_count[1]; // step
	out_buf->wrote_len += 6;
}

void compile_set_lfo_wave(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs)
{
	check_outbuf_remain_size(3, out_buf->wrote_len);

	in_buf->idx += 2;
	check_eof_exit_macro(in_buf, cs);

	const int8_t channel = get_lfo_channel(in_buf->buf[(in_buf->idx)++]);
	if(channel < 0) exit_mml_one_back(MMLEC_UNEXPECTED_CHAR);
	skip_blank_check_eof_exit_macro(in_buf, cs);

	StrProcErrorCode sec = str_to_num_u8(in_buf, &tmp_u8_static, 0x00, N_LFO_WAVE_TYPE - 1);
	if(sec != SEC_NO_ERROR) exit_str_proc(sec);

	// ߏo.
	out_buf->buf[out_buf->wrote_len + 0] = OP_LFO_SET_WAVE;
	out_buf->buf[out_buf->wrote_len + 1] = (uint8_t)channel;
	out_buf->buf[out_buf->wrote_len + 2] = tmp_u8_static; // wave
	out_buf->wrote_len += 3;
}

void compile_lfo_switch(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs)
{
	check_outbuf_remain_size(3, out_buf->wrote_len);

	in_buf->idx += 2;
	check_eof_exit_macro(in_buf, cs);

	const int8_t channel = get_lfo_channel(in_buf->buf[(in_buf->idx)++]);
	if(channel < 0) exit_mml_one_back(MMLEC_UNEXPECTED_CHAR);
	skip_blank_check_eof_exit_macro(in_buf, cs);

	StrProcErrorCode sec = str_to_num_u8(in_buf, &tmp_u8_static, 0x00, 0x07);
	if(sec != SEC_NO_ERROR) exit_str_proc(sec);

	// ߏo.
	out_buf->buf[out_buf->wrote_len + 0] = OP_LFO_SWITCH;
	out_buf->buf[out_buf->wrote_len + 1] = (uint8_t)channel;
	out_buf->buf[out_buf->wrote_len + 2] = tmp_u8_static; // mode
	out_buf->wrote_len += 3;
}

void compile_set_lfo_operators(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs)
{
	if(cs->header->source_type != SRC_TYPE_OPL3) exit_mml(MMLEC_UNSUPPORTED_SOURCE); //  OPL3 ̂ݑΉ.
	check_outbuf_remain_size(3, out_buf->wrote_len);

	in_buf->idx += 2;
	check_eof_exit_macro(in_buf, cs);

	const int8_t channel = get_lfo_channel(in_buf->buf[(in_buf->idx)++]);
	if(channel < 0) exit_mml_one_back(MMLEC_UNEXPECTED_CHAR);
	skip_blank_check_eof_exit_macro(in_buf, cs);

	// ݂̉F 2OP 0`3͈̔́A4OP 0`15͈̔͂ꂼ󂯕t.
	const uint8_t max_val = (cs->part_status.flag_specific_0 & PFLAG_OPL3_0_4OP) ? 0x0f : 0x03;
	StrProcErrorCode sec = str_to_num_u8(in_buf, &tmp_u8_static, 0x00, max_val);
	if(sec != SEC_NO_ERROR) exit_str_proc(sec);

	out_buf->buf[out_buf->wrote_len + 0] = OP_SET_LFO_OP;
	out_buf->buf[out_buf->wrote_len + 1] = (uint8_t)channel;
	out_buf->buf[out_buf->wrote_len + 2] = tmp_u8_static; // bit set
	out_buf->wrote_len += 3;
}

void compile_set_hw_lfo_params(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs)
{
	if(cs->header->source_type != SRC_TYPE_OPL3) exit_mml(MMLEC_UNSUPPORTED_SOURCE); //  OPL3 ̂ݑΉ.
	check_outbuf_remain_size(2, out_buf->wrote_len);

	in_buf->idx += 2;
	check_eof_exit_macro(in_buf, cs);

	StrProcErrorCode sec = str_to_num_u8(in_buf, &tmp_u8_static, 0x00, 0x03); // 2rbg (DAM,DVB) w肷.
	if(sec != SEC_NO_ERROR) exit_str_proc(sec);

	// ߏo.
	out_buf->buf[out_buf->wrote_len + 0] = OP_HW_LFO_SET_PARAM;
	out_buf->buf[out_buf->wrote_len + 1] = (tmp_u8_static << 6); // OɃVtgĂ.
	out_buf->wrote_len += 2;
}

void compile_set_panpot(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs)
{
	if(cs->header->source_type != SRC_TYPE_OPL3) exit_mml(MMLEC_UNSUPPORTED_SOURCE); //  OPL3 ̂ݑΉ.
	check_outbuf_remain_size(2, out_buf->wrote_len);

	in_buf->idx += 1;
	check_eof_exit_macro(in_buf, cs);

	StrProcErrorCode sec = str_to_num_u8(in_buf, &tmp_u8_static, 1, 3);
	if(sec != SEC_NO_ERROR) exit_str_proc(sec);

	// ߏo.
	out_buf->buf[out_buf->wrote_len + 0] = OP_SET_PANPOT;
	out_buf->buf[out_buf->wrote_len + 1] = (tmp_u8_static << 4); // OɃVtgĂ.
	out_buf->wrote_len += 2;
}
void compile_portament(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs)
{
	if(cs->header->source_type != SRC_TYPE_OPL3) exit_mml(MMLEC_UNSUPPORTED_SOURCE); //  OPL3 ̂ݑΉ.
	if(cs->part_status.prev_note_pos == 0 || out_buf->buf[cs->part_status.prev_note_pos] == OP_REST)
	{
		// Ox̏ꍇu߂ȂvG[ɓZ߂.
		extern const char ___FAR msg_no_prev_note[];
		set_error_info_addnl_msg(msg_no_prev_note);
		exit_mml(MMLEC_INVALID_PORTAMENT);
	}
	if((cs->part_status.flag_common_0 & PFLAG_COMM_0_PORT_OK) == 0)
	{
		extern const char ___FAR msg_continuous_portament[];
		set_error_info_addnl_msg(msg_continuous_portament);
		exit_mml(MMLEC_INVALID_PORTAMENT);
	}

	in_buf->idx += 1;
	skip_blank_check_eof_exit_macro(in_buf, cs);

	// O̔߂̎OɃ|^gߗp̗̈mۂ.
	check_outbuf_remain_size(6, out_buf->wrote_len); // 6oCgo͂]TKv.
	uint16_t idx = out_buf->wrote_len - 1;
	for(; cs->part_status.prev_note_pos <= idx; --idx) { out_buf->buf[idx + 6] = out_buf->buf[idx]; }
	out_buf->wrote_len += 6;
	cs->part_status.prev_note_pos += 6;
	const uint16_t pbpos = idx + 1; // for 1]vɌĂ.

	// ڕW̉擾.
	proc_consecutive_octave_update(in_buf, out_buf, cs);
	get_note_num(in_buf, &tmp_u8_static, cs);

	// |^g߂̃p[^vZ.
	const int32_t d_sent = calc_sent_diff(keyon_op_to_note_num(out_buf->buf[cs->part_status.prev_note_pos]), tmp_u8_static); // sent Pʂł̕ω.
	const int32_t d_sent_shifted = d_sent << 18; // i32  sent Ȃő̃Vtg(Zɐxۂ).
	const int32_t d_per_clock = div_i32(d_sent_shifted, out_buf->buf[pbpos + 1]); // Pʕω.
	const int32_t d_fixed_point = ar_shift(d_per_clock, 2); // Pʕωʂ [31:16].[15:0] ̌Œ菬_ɕϊ.

	out_buf->buf[pbpos] = OP_PORTAMENT;
	out_buf->buf[pbpos + 1] = out_buf->buf[cs->part_status.prev_note_pos + 1]; // |^g̒͒O̔߂̒.
	*((int32_t ___FAR*)(out_buf->buf + pbpos + 2)) = d_fixed_point;

	cs->part_status.flag_common_0 &= ~(PFLAG_COMM_0_PORT_OK);
}

void compile_loop(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs)
{
	check_outbuf_remain_size(1, out_buf->wrote_len);
	in_buf->idx += 1;
	out_buf->buf[out_buf->wrote_len] = OP_LOOP_START_POS;
	out_buf->wrote_len += 1;

	cs->part_status.flag_common_0 |= PFLAG_COMM_0_INF_LOOP;
}

void compile_set_tempo_n(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs)
{
	in_buf->idx += 2;
	skip_blank_check_eof_exit_macro(in_buf, cs);
	const StrProcErrorCode sec = str_to_num_u8(in_buf, &tmp_u8_static, 1, 255);
	if(sec != SEC_NO_ERROR) exit_str_proc(sec);
	tmp_u8_static = convert_tempo_to_timer_b(tmp_u8_static, cs->zenlen); // e| timer_b ̒lɕϊ.
	output_set_tempo(out_buf, tmp_u8_static);
}

static void compile_set_tempo(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs)
{
	const char ch = in_buf->buf[(in_buf->idx)++];
	skip_blank_check_eof_exit_macro(in_buf, cs);

	const size_t num_pos = in_buf->idx;
	const StrProcErrorCode sec = str_to_num_u8(in_buf, &tmp_u8_static, 0, 255);
	if(sec != SEC_NO_ERROR) exit_str_proc(sec);

	if(ch == 't')
	{
		if(tmp_u8_static == 0) // t 0w͂łȂ.
			exit_mml_idx(MMLEC_NUM_SMALL, num_pos);

		// 48NbNPʂ BPM  timer_b ɕϊ.
		if (tmp_u8_static < (256 - TBL_SZ_48CLK_TO_TIMER_B)) tmp_u8_static = 0;
		else tmp_u8_static = TBL_48CLK_TO_TIMER_B[tmp_u8_static - (256 - TBL_SZ_48CLK_TO_TIMER_B)];
	}

	output_set_tempo(out_buf, tmp_u8_static);
}

static void output_set_tempo(OutputBuffer* out_buf, uint8_t timer_b_val)
{
	// o̓obt@TCY`FbN͋ʂłōsĂ.
	check_outbuf_remain_size(15, out_buf->wrote_len);

	// ߏo.
	out_buf->buf[out_buf->wrote_len + 0] = OP_SET_TIMER_B;
	out_buf->buf[out_buf->wrote_len + 1] = timer_b_val;
	out_buf->buf[out_buf->wrote_len + 2] = 0;
	*((uint16_t ___FAR*)(out_buf->buf + out_buf->wrote_len + 3)) = convert_timer_b_to_8253a(timer_b_val, 0);
	out_buf->buf[out_buf->wrote_len + 5] = 0;
	*((uint16_t ___FAR*)(out_buf->buf + out_buf->wrote_len + 6)) = convert_timer_b_to_8253a(timer_b_val, 1);
	out_buf->buf[out_buf->wrote_len + 8] = 0;
	*((uint16_t ___FAR*)(out_buf->buf + out_buf->wrote_len + 9)) = convert_timer_b_to_8253a(timer_b_val, 2);
	out_buf->buf[out_buf->wrote_len + 11] = 0;
	out_buf->buf[out_buf->wrote_len + 12] = TBL_TIMER_B_TO_SB_TIME_CONSTANT[timer_b_val];
	out_buf->buf[out_buf->wrote_len + 13] = TBL_TIMER_B_TO_SB_DMA_CNT[timer_b_val];
	out_buf->buf[out_buf->wrote_len + 14] = TBL_TIMER_B_TO_SB_N_WAIT[timer_b_val];
	out_buf->wrote_len += 15;
}

static void compile_fnum_blk(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs)
{
	// HS ߂ H ߂.
	uint8_t is_HS = 0;
	if(((in_buf->idx + 1) < in_buf->read_len) && in_buf->buf[in_buf->idx + 1] == 'S')
		is_HS = 1;
	check_outbuf_remain_size((is_HS ? 4 : 2), out_buf->wrote_len);

	in_buf->idx += (is_HS ? 2 : 1);
	check_eof_exit_macro(in_buf, cs);

	const int8_t setting_idx = get_fnum_blk_setting_idx(in_buf->buf[(in_buf->idx)++]);
	if(setting_idx < 0) exit_mml_one_back(MMLEC_UNEXPECTED_CHAR);

	if(is_HS)
	{
		skip_blank_check_eof_exit_macro(in_buf, cs);

		// FNum ̒l.
		StrProcErrorCode sec = str_to_num_u32(in_buf, &tmp_u32_static, 0, OPL3_FNUM_MAX);
		if(sec != SEC_NO_ERROR) exit_str_proc(sec);
		const uint16_t fnum = (uint16_t)tmp_u32_static;
		skip_one_comma(in_buf, cs, 1);

		// BLK ̒l.
		sec = str_to_num_u8(in_buf, &tmp_u8_static, 0, 7);
		if(sec != SEC_NO_ERROR) exit_str_proc(sec);

		// ߏo (FNum, BLK ̓gGfBAŋL^).
		out_buf->buf[out_buf->wrote_len + 0] = OP_SET_FNUM_BLK;
		out_buf->buf[out_buf->wrote_len + 1] = (uint8_t)setting_idx;
		out_buf->buf[out_buf->wrote_len + 2] = (uint8_t)(fnum & 0xff);
		out_buf->buf[out_buf->wrote_len + 3] = (uint8_t)((fnum >> 8) | (tmp_u8_static << 2));
		out_buf->wrote_len += 4;
	}
	else
	{
		skip_blank_exit_macro(in_buf, cs); // EOF ĂG[łȂ(ftHgg)
		get_note_length(in_buf, &tmp_u8_static, cs);

		cs->part_status.prev_note_pos = out_buf->wrote_len;

		out_buf->buf[out_buf->wrote_len + 0] = OP_PLAY_FNUM_BLK_0 + (uint8_t)setting_idx;
		out_buf->buf[out_buf->wrote_len + 1] = tmp_u8_static - 1; // note_len
		out_buf->wrote_len += 2;

		cs->part_status.flag_common_0 &= ~(PFLAG_COMM_0_INF_LOOP);
		add_clock_buf(cs, tmp_u8_static);
	}
}

static void compile_set_scale(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs)
{
	(void)out_buf;

	in_buf->idx += 2;
	// ݒJn̊ʂ`FbN(u@=(v̊Ԃɋ󔒂͋Ȃ).
	if(in_buf->read_len <= in_buf->idx) exit_mml(MMLEC_REACH_EOF);
	if (in_buf->buf[in_buf->idx] != '(') exit_mml(MMLEC_UNEXPECTED_CHAR);
	in_buf->idx += 1;

	// ݒI̊ʂoĂ܂ňڒ̐ݒǂ.
	uint8_t first_run = 1, has_comma;
	char ch;
	while(1) {
		if(first_run)
		{
			skip_blank_check_eof_exit_macro(in_buf, cs);
			ch = in_buf->buf[(in_buf->idx)++];
			if(ch == ')') break; // ʂłG[ɂȂ.
			has_comma = 1; // J}邱Ƃɂ.
		}
		else
		{
			has_comma = skip_one_comma(in_buf, cs, 0);
			ch = in_buf->buf[(in_buf->idx)++];
			if(!has_comma && ch == ')') break;
		}

		uint8_t sidx = 0;
		switch (ch)
		{
			case 'c': { sidx = 0; break; }
			case 'd': { sidx = 1; break; }
			case 'e': { sidx = 2; break; }
			case 'f': { sidx = 3; break; }
			case 'g': { sidx = 4; break; }
			case 'a': { sidx = 5; break; }
			case 'b': { sidx = 6; break; }
			default: {
				// [UreB̂߃G[bZ[W𕪂.
				if (has_comma) exit_mml_one_back(MMLEC_NOT_NOTE_OP_CH);
				else exit_mml_one_back(MMLEC_UNEXPECTED_CHAR);
			}
		}
		// J} c-b oĂG[.
		if(!has_comma) exit_mml_one_back(MMLEC_UNEXPECTED_CHAR);

		// +, -, = ǂꂩ.
		skip_blank_exit_macro(in_buf, cs);
		ch = in_buf->buf[(in_buf->idx)++];
		if(ch != '+' && ch != '-' && ch != '=')
		{
			extern const char ___FAR msg_set_scale_no_change[];
			set_error_info_addnl_msg(msg_set_scale_no_change);
			exit_mml_one_back(MMLEC_INVALID_SET_SCALE);
		}

		skip_blank_exit_macro(in_buf, cs);
		int8_t diff;
		if(ch == '=') diff = 0;
		else
		{
			// lw`FbN(l +1  -1 ̕ωʂƂĈ)
			const size_t orig_pos = in_buf->idx;
			diff = (ch == '+') ? 1 : -1;
			StrProcErrorCode sec = str_to_num_u8(in_buf, &tmp_u8_static, 0, 128);
			if(sec == SEC_NO_ERROR)
			{
				if(tmp_u8_static == 128)
				{
					if(ch == '-') diff = -128;
					else
					{
						in_buf->idx = orig_pos;
						exit_str_proc(SEC_NUM_LARGE); // + w 128 ͔͈͊O.
					}
				} else diff *= (int8_t)tmp_u8_static;
			}
			else if(sec != SEC_NOT_NUM) exit_str_proc(sec);
		}

		cs->part_status.scale_setting[sidx] = diff;
		first_run = 0;
	}
}

uint8_t convert_tempo_to_timer_b(const uint8_t tempo, const uint8_t zenlen)
{
	// tempo  zenlen g timer_b lvZ.
	// e|vZ 240* 3993600 / (zenlen * 1152 * tempo) ł邪ȀZ̔폜Ə͗Ƃ uint32_t ̍ől𒴂邱Ƃ͂Ȃ.
	// 240 * 3993600 = 958,464,000 < 4,294,967,295
	// 255 * 1152 * 255 = 74,908,800 < 4,294,967,295
	const uint32_t dividend = 240 * 3993600;
	const uint32_t divisor = mul_u32(zenlen, mul_u32(1152, tempo));
	uint32_t tb = div_u32(dividend, divisor);
	const uint32_t amari = mod_u32(dividend, divisor);
	if((divisor / 2 < amari) && (tb != 0xffffffff)) tb += 1; // ܂g̎ľܓ( <= łȂ < ɂȂĂ邩͌؃R[hQƂ̂)
	if(tb < 1) tb = 1;
	if(256 < tb) tb = 256;
	tb = 256 - tb;
	return (uint8_t)tb;
}

uint16_t convert_timer_b_to_8253a(const uint8_t timer_b, const uint8_t type)
{
	// timer_b ̒l 8253A ̒lɕϊ(̓e[u).
	// (type == 0) ̏ꍇ 5,10MHz nA(type == 1) ̏ꍇ 8MHz nAȊȌꍇ PC/AT ݊@p̒l߂.
	if(type == 0) {
		if(timer_b < (256 - TBL_SZ_TIMER_B_TO_8253A_5MHz)) return 0xffff;
		else return TBL_TIMER_B_TO_8253A_5MHz[timer_b - (256 - TBL_SZ_TIMER_B_TO_8253A_5MHz)];
	}
	else if(type == 1)
	{
		if(timer_b < (256 - TBL_SZ_TIMER_B_TO_8253A_8MHz)) return 0xffff;
		else return TBL_TIMER_B_TO_8253A_8MHz[timer_b - (256 - TBL_SZ_TIMER_B_TO_8253A_8MHz)];
	}
	else
	{
		if(timer_b < (256 - TBL_SZ_TIMER_B_TO_8253A_PCAT)) return 0xffff;
		else return TBL_TIMER_B_TO_8253A_PCAT[timer_b - (256 - TBL_SZ_TIMER_B_TO_8253A_PCAT)];
	}
}

void expand_macro(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs)
{
	(void)out_buf; //unused
	const size_t start = in_buf->idx;

	// $ ƃ}N̊Ԃɋ󔒂݂͑łȂ.
	in_buf->idx += 1;
	check_eof_exit_macro(in_buf, cs);

	// v}NT.
	const size_t remain_len = in_buf->read_len - in_buf->idx;
	uint8_t macro_found = 0;
	const Macro ___FAR* m = macros;
	for(uint8_t macro_idx = 0; macro_idx < n_defined_macros; ++macro_idx)
	{
		const size_t name_len = m->name_len;

		// }N͒Ƀ\[gĂ̂ŁAcobt@}NZ break
		if(remain_len < name_len) break;

		if(strncmp(in_buf->buf + in_buf->idx, in_buf->buf + m->name_idx, name_len) == 0)
		{
			macro_found = 1;
			break; // v}N break
		}
		m += 1;
	}
	if(!macro_found)
	{
		extern const char ___FAR msg_macro_undefined[];
		set_error_info_addnl_msg(msg_macro_undefined);
		exit_mml_idx(MMLEC_MACRO_EXPAND, start); // }NȂG[.
	}

	// in_buf->idx }N̐܂Ői߂āAin_buf }Nɒu.
	in_buf->idx += m->name_len;
	if(MACRO_NEST_MAX - 1 <= cs->part_status.n_macro_nest)
	{
		extern const char ___FAR msg_macro_nest_max[];
		set_error_info_addnl_msg(msg_macro_nest_max);
		exit_mml_idx(MMLEC_MACRO_EXPAND, start); // }ÑlXg񐔑G[.
	}
	cs->part_status.macro_nest_stack[cs->part_status.n_macro_nest] = *in_buf;
	cs->part_status.n_macro_nest += 1;
	in_buf->idx = m->body_idx;
	in_buf->read_len = m->body_idx + m->body_len;
}

void modify_note_length(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs)
{
	if(cs->part_status.prev_note_pos == 0)
	{
		extern const char ___FAR msg_no_prev_note[];
		set_error_info_addnl_msg(msg_no_prev_note);
		exit_mml(MMLEC_INVALID_MOD_LEN);
	}

	in_buf->idx += 1;
	skip_blank_check_eof_exit_macro(in_buf, cs);
	const size_t num_start = in_buf->idx;

	// }CiX邩mF.
	const uint8_t is_minus = (in_buf->buf[in_buf->idx] == '-');
	if(is_minus)
	{
		in_buf->idx += 1;
		skip_blank_check_eof_exit_macro(in_buf, cs);
	}

	MMLErrorCode mec = get_note_length(in_buf, &tmp_u8_static, cs);
	if(mec == MMLEC_NOT_NUM) exit_mml(mec);

	// Oɏo͂ߋyыx̉𒲐.
	uint8_t ___FAR* p_prev_len = out_buf->buf + cs->part_status.prev_note_pos + 1;
	const uint8_t prev_len = *p_prev_len;
	if(!is_minus)
	{
		if(0xff - prev_len < tmp_u8_static)
		{
			extern const char ___FAR msg_len_cannot_add[];
			set_error_info_addnl_msg(msg_len_cannot_add);
			exit_mml_idx(MMLEC_INVALID_NOTE_LEN, num_start); // ȏ㉹𒷂łȂ.
		}
		*p_prev_len += tmp_u8_static;
		add_clock_buf(cs, tmp_u8_static);
	}
	else
	{
		if(prev_len <= tmp_u8_static)
		{
			extern const char ___FAR msg_len_cannot_sub[];
			set_error_info_addnl_msg(msg_len_cannot_sub);
			exit_mml_idx(MMLEC_INVALID_NOTE_LEN, num_start); // ȏ㉹ZłȂ.
		}
		*p_prev_len -= tmp_u8_static;
		sub_clock_buf(cs, tmp_u8_static);
	}
}

void update_note_length(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs)
{
	(void)out_buf; // unused

	in_buf->idx += 1;
	skip_blank_check_eof_exit_macro(in_buf, cs);

	MMLErrorCode mec = get_note_length(in_buf, &tmp_u8_static, cs);
	if(mec == MMLEC_NOT_NUM) exit_mml(mec);

	cs->part_status.note_length = tmp_u8_static; // new_len
}

void process_comment(InputBuffer* in_buf, OutputBuffer *out_buf, CompileStatus* cs)
{
	(void)out_buf; // unused
	(void)cs;
	move_to_EOL(in_buf);
}

void process_part_filter(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs)
{
	(void)out_buf; // unused

	in_buf->idx += 1;
	exit_macro(in_buf, cs);

	if(in_buf->read_len <= in_buf->idx) return; // u|v̒オ EOF Ȃ̂̓G[ɂȂ.
	if(is_whitespace(in_buf->buf[in_buf->idx])) return; // u|v̌Ƀp[gw肪ȂΑSp[gRpCĊJ.

	// p[gw͂ǂ̃p[gŃRpCΏۂɂȂ邩\łȂ̂Ŏ~ނ𓾂 check_part_ch ͏ 1 ɂ.
	const uint8_t no_skip = current_part_is_exist(in_buf, cs, 1);
	char ch;
	// 󔒂s܂Ŕ΂.
	while(in_buf->idx < in_buf->read_len)
	{
		ch = in_buf->buf[in_buf->idx];
		if(is_newline(ch)) return; // s΃XLbv̕Kv͂Ȃ̂őI.
		in_buf->idx += 1;
		if(is_blank(ch)) break;
	}

	//  in_buf->idx ́u|v̎ɏo󔒂̎̕܂ EOF
	if(no_skip) return;
	while(in_buf->idx < in_buf->read_len)
	{
		ch = in_buf->buf[in_buf->idx];
		if(is_newline(ch) || ch == '|') break; // ch == '|' ł̂܂ܔčēx̊֐ɏ.
		in_buf->idx += 1;
		if(in_buf->idx < in_buf->read_len) exit_macro(in_buf, cs); // EOF ł}NȂ.
	}
}

MMLErrorCode get_note_length(InputBuffer* in_buf, uint8_t* note_len_by_clock, CompileStatus* cs)
{
	static uint8_t tmp_my_static; // note_len_by_clock  tmp_u8_static ł\邽߁AƎ static ϐpӂĎg.

	// Ou % ΃NbNڎwƂĈ.
	if(in_buf->buf[in_buf->idx] == '%')
	{
		in_buf->idx += 1;
		skip_blank_check_eof_exit_macro(in_buf, cs);

		StrProcErrorCode ret = str_to_num_u8(in_buf, &tmp_my_static, 1, 255); // ŏ1
		if(ret != SEC_NO_ERROR) exit_mml(ec_convert_str_to_mml(ret)); // Kl݂锤.
		*note_len_by_clock = tmp_my_static;
		return MMLEC_NO_ERROR;
	}

	// ŏ1B0͖(00ZɂȂ邵ANbNw0NbN炷Ƃ̂)
	StrProcErrorCode ret = str_to_num_u8(in_buf, &tmp_my_static, 1, 255);
	if(ret == SEC_NO_ERROR)
	{
		if(cs->zenlen % tmp_my_static != 0) exit_mml(MMLEC_INVALID_NOTE_LEN);
		tmp_my_static = cs->zenlen / tmp_my_static;
		*note_len_by_clock = tmp_my_static;
	}
	else
	{
		if(ret != SEC_NOT_NUM) exit_mml(ec_convert_str_to_mml(ret));

		*note_len_by_clock = cs->part_status.note_length; // ftHgݒ.
		tmp_my_static = cs->part_status.note_length;
		if(!(in_buf->idx < in_buf->read_len && in_buf->buf[in_buf->idx] == '.')) return MMLEC_NOT_NUM; // t_Ȃ΂ŏI.
	}

	// t_̃`FbN.
	while(in_buf->idx < in_buf->read_len && in_buf->buf[in_buf->idx] == '.')
	{
		if(tmp_my_static % 2 == 1) exit_mml(MMLEC_TOO_MUCH_DOTS);
		tmp_my_static /= 2;
		*note_len_by_clock += tmp_my_static;
		in_buf->idx += 1;
	}

	//  EOF BĂt_ȂŃG[ł͂Ȃ.
	return MMLEC_NO_ERROR;
}

MMLErrorCode get_note_num(InputBuffer* in_buf, uint8_t* note_num, CompileStatus* cs)
{
	int16_t ret = 0;
	uint8_t sidx = 0;
	const char note_ch = in_buf->buf[(in_buf->idx)++];
	switch(note_ch)
	{
		case 'c': { ret = 0; sidx = 0; break; }
		case 'd': { ret = 2; sidx = 1; break; }
		case 'e': { ret = 4; sidx = 2; break; }
		case 'f': { ret = 5; sidx = 3; break; }
		case 'g': { ret = 7; sidx = 4; break; }
		case 'a': { ret = 9; sidx = 5; break; }
		case 'b': { ret = 11; sidx = 6; break; }
		default: exit_mml_one_back(MMLEC_NOT_NOTE_OP_CH);
	}
	ret += (cs->part_status.octave - 1) * 12; // IN^[u 1`8 ŎĂ̂1.

	// ՎL̔.
	// ՎLȂ΃XP[w̕ωʂKp.
	skip_blank_exit_macro(in_buf, cs);
	const char acc_ch = in_buf->buf[in_buf->idx];
	if(acc_ch == '=') in_buf->idx += 1; // i`Ȃ牽Ȃ.
	else if(acc_ch == '#' || acc_ch == '_')
	{
		ret += (acc_ch == '#') ? 1 : -1;
		in_buf->idx += 1;
		skip_blank_exit_macro(in_buf, cs);
		if(in_buf->buf[in_buf->idx] == acc_ch) // _uV[vA_utbg.
		{
			ret += (acc_ch == '#') ? 1 : -1;
			in_buf->idx += 1;
		}
	}
	else ret += cs->part_status.scale_setting[sidx];

	// グ.
	// r int16_t ͈̔͂悤Ƃ痬΂ɂŒ߂ăG[ɂ.
	while(1)
	{
		skip_blank_exit_macro(in_buf, cs);

		const char ch = in_buf->buf[in_buf->idx];
		if(ch != '+' && ch != '-') break;
		if(ch == '+')
		{
			if(ret == INT16_MAX) exit_mml(MMLEC_NOTE_NUM_OF_RANGE);
			ret += 1;
		}
		else // if(ch == '-')
		{
			if(ret == INT16_MIN) exit_mml(MMLEC_NOTE_NUM_OF_RANGE);
			ret -= 1;
		}
		in_buf->idx += 1;
	}

	// hCodlIȔ͈͓mF.
	if(ret < NOTE_NUMBER_MIN || NOTE_NUMBER_MAX < ret)
		exit_mml(MMLEC_NOTE_NUM_OF_RANGE);

	*note_num = (uint8_t)ret;
	return MMLEC_NO_ERROR;
}

uint8_t note_num_to_keyon_op(const uint8_t note_num)
{
	const uint8_t octave = note_num / 12;
	const uint8_t n = note_num % 12;
	return 0x70 + (octave * 0x10) + n;
}

uint8_t keyon_op_to_note_num(const uint8_t keyon_op)
{
	const uint8_t tmp = keyon_op - 0x70;
	return ((tmp / 16) * 12) + (tmp % 16);
}

int32_t calc_sent_diff(const uint8_t note_num_a, const uint8_t note_num_b)
{
	// 12 K sent ̒lɕϊe[u.
	static const uint16_t sent_table_opl3[12] = { 0, 85, 171, 256, 341, 427, 512, 597, 683, 768, 853, 939 };

	// (傫ق-ق)߂.
	const uint8_t b_is_larger = note_num_a < note_num_b;
	const uint8_t nn_large = b_is_larger ? note_num_b : note_num_a;
	const uint8_t nn_small = b_is_larger ? note_num_a : note_num_b;

	// IN^[uPʂ̍.
	const uint8_t d_oct = (nn_large - nn_small) / 12;
	//const uint8_t d_note = (nn_large - nn_small) % 12; // ͎gȂ.
	int32_t ret = 1024 * d_oct;

	// IN^[u̍.
	const uint8_t r_large = nn_large % 12;
	const uint8_t r_small = nn_small % 12;
	if(r_small <= r_large) ret += sent_table_opl3[r_large] - sent_table_opl3[r_small];
	else ret += sent_table_opl3[r_large] + (1024 - sent_table_opl3[r_small]);

	if(!b_is_larger) ret = -ret;

	return ret;
}

uint8_t proc_consecutive_octave_update(InputBuffer* in_buf, OutputBuffer* out_buf, CompileStatus* cs)
{
	// AIN^[uύX߂߂.
	// IN^[u߂ǂtOŕԂ.
	uint8_t ret = 0;
	while(1)
	{
		const char ch = in_buf->buf[in_buf->idx];
		if(!(ch == '<' || ch == '>' || ch == 'o')) break;
		update_octave(in_buf, out_buf, cs);
		skip_blank_check_eof_exit_macro(in_buf, cs);
		ret = 1;
	}
	return ret;
}

void exit_str_proc(const StrProcErrorCode sec)
{
	// ݂͌̓ǂݏoʒuG[̈ʒu.
	switch(sec)
	{
	case SEC_EOF:
	case SEC_NOT_NUM:
	case SEC_UNEXPECTED_CHAR:
	case SEC_NUM_LARGE:
	case SEC_NUM_SMALL:
		exit_mml(ec_convert_str_to_mml(sec));
		break;
	default: {}
	}

	// G[̈ʒuǂݏoʒu1ił.
	exit_mml_one_back(ec_convert_str_to_mml(sec));
}

uint8_t is_pitch_name_char(const char c)
{
	if(c == 'c' || c == 'd' || c == 'e' || c == 'f' || c == 'g' || c == 'a' || c == 'b') return 1;
	return 0;
}

int8_t get_lfo_channel(const char c)
{
	if(c == 'a') return 0;
	else if(c == 'b') return 1;
	else if(c == 'c') return 2;
	return -1;
}

int8_t get_fnum_blk_setting_idx(const char c)
{
	if(c == 'a') return 0;
	else if(c == 'b') return 1;
	else if(c == 'c') return 2;
	else if(c == 'd') return 3;
	return -1;
}

// skip blank check eof exit macro n֐̎.
static void sb_ce_ex_impl(InputBuffer* in_buf, CompileStatus* cs, uint8_t do_skip, uint8_t eof_error)
{
	while(1)
	{
		if(do_skip) str_skip(in_buf, SST_SKIP_BLANK);

		if(in_buf->idx < in_buf->read_len) break; // EOF ɓBĂȂ.

		if(cs->part_status.n_macro_nest == 0)
		{
			// t@C EOF ɓBꍇ.
			if(eof_error) exit_mml(MMLEC_REACH_EOF);
			else break;
		}

		// }NI[ȂA}N1iKEo.
		//  while ɖ߂̂ŁA(󔒂΂ɍ)EOFłȂꏊɏo邩 n_macro_nest 0ɂȂ܂Ń}NEoJԂ.
		const uint8_t idx = --(cs->part_status.n_macro_nest);
		in_buf->idx = cs->part_status.macro_nest_stack[idx].idx;
		in_buf->read_len = cs->part_status.macro_nest_stack[idx].read_len;
	}
}

// 󔒂XLbvāAEOF ɓB΃G[ɂāA}NI[ȂEo.
void skip_blank_check_eof_exit_macro(InputBuffer* in_buf, CompileStatus* cs)
{
	sb_ce_ex_impl(in_buf, cs, 1, 1);
}

// 󔒂XLbvāAEOF ɓBĂG[ɂA}NI[ȂEo.
void skip_blank_exit_macro(InputBuffer* in_buf, CompileStatus* cs)
{
	sb_ce_ex_impl(in_buf, cs, 1, 0);
}

// 󔒂XLbvAEOF ɓBĂG[ɂāA}NI[ȂEo.
void check_eof_exit_macro(InputBuffer* in_buf, CompileStatus* cs)
{
	sb_ce_ex_impl(in_buf, cs, 0, 1);
}

// 󔒂XLbvAEOF ɓBĂĂG[ɂA}NI[ȂEo.
void exit_macro(InputBuffer* in_buf, CompileStatus* cs)
{
	sb_ce_ex_impl(in_buf, cs, 0, 0);
}

uint8_t skip_one_comma(InputBuffer* in_buf, CompileStatus* cs, const uint8_t fail_on_err)
{
	skip_blank_exit_macro(in_buf, cs);

	const uint8_t is_eof = (in_buf->read_len <= in_buf->idx);
	const uint8_t is_comma = is_eof ? 0 : (in_buf->buf[in_buf->idx] == ',');

	if(is_eof || !is_comma)
	{
		// J} EOF
		if(fail_on_err) exit_mml(MMLEC_UNEXPECTED_CHAR);
		return 0;
	}

	// J}L.
	in_buf->idx += 1;
	skip_blank_exit_macro(in_buf, cs);

	return 1;
}

void check_outbuf_remain_size(size_t n_needed, size_t wrote_len)
{
	if(MAX_OUT_SIZE - n_needed < wrote_len)
		exit_mml_op_start_idx(MMLEC_OUT_SIZE_LIMIT);
}

static void add_clock_buf(CompileStatus* cs, const uint16_t clk)
{
	const uint8_t idx = cs->part_status.n_repeat_nest;
	const uint8_t mask = (1 << (idx - 1));
	if(cs->part_status.repeat_esc_flag & mask)
		cs->part_status.clock_buf_esc[idx] += clk;
	else
		cs->part_status.clock_buf[idx] += clk;
}

static void sub_clock_buf(CompileStatus* cs, const uint16_t clk)
{
	const uint8_t idx = cs->part_status.n_repeat_nest;
	const uint8_t mask = (1 << (idx - 1));
	if(cs->part_status.repeat_esc_flag & mask)
		cs->part_status.clock_buf_esc[idx] -= clk;
	else
		cs->part_status.clock_buf[idx] -= clk;
}


// MML ̃RpCȊÕR[h.
void init_compiler()
{
	// 邱Ƃ.
}

const char ___FAR* mml_error_code_to_str(const MMLErrorCode error_code)
{
	extern const char ___FAR msg_unknown_mml_error_code[]; // strs.c
	extern const char ___FAR* ___FAR ERROR_STRINGS[];
	if(N_MML_ERROR_CODES <= error_code) return msg_unknown_mml_error_code;
	else return ERROR_STRINGS[error_code];
}

MMLErrorCode ec_convert_str_to_mml(const StrProcErrorCode error_code)
{
	switch(error_code)
	{
	case SEC_NO_ERROR:			return MMLEC_NO_ERROR;
	case SEC_EOF:				return MMLEC_REACH_EOF;
	case SEC_NOT_NUM:			return MMLEC_NOT_NUM;
	case SEC_NUM_SMALL:			return MMLEC_NUM_SMALL;
	case SEC_NUM_LARGE:			return MMLEC_NUM_LARGE;
	case SEC_UNEXPECTED_CHAR:	return MMLEC_UNEXPECTED_CHAR;
	case SEC_STRING_NOT_FOUND: // 񂪌Ȃ(̃G[ɑΉ MML ̃G[͓sxK؂Ȃ̂̏ŋ߂)
	default:
		return MMLEC_UNKNOWN_ERROR; // ͂肦Ȃ.
	}
}
