texture tex0;

sampler smp = sampler_state
{
	Texture = <tex0>;
	MinFilter = LINEAR;
	MagFilter = LINEAR;
	MipFilter = NONE;
	AddressU = Clamp;
	AddressV = Clamp;
};

struct VS_Output
{
	float4 pos : POSITION;
	float2 tex : TEXCOORD0;
};

uniform float2 tex0sz;
uniform float2 outsz;

VS_Output vs(float4 pos : POSITION,
             float4 col : COLOR0,
             float2 tex : TEXCOORD0)
{
	VS_Output o;
	o.pos = pos;
	o.tex = tex;
	return o;
}

// シェーダ用乱数ルーチンらしい(0.0f～1.0f).
// https://stackoverflow.com/questions/4200224/random-noise-functions-for-glsl
float rand(float2 pos)
{
	return frac(sin(dot(pos.xy, float2(12.9898, 78.233))) * 43756.5453);
}

float4 gaussian_sample(float2 uv, float2 dx, float2 dy)
{
	float4 col = 0.0f;
	col += tex2D(smp, uv - dx - dy) * 1.0f / 16.0f;
	col += tex2D(smp, uv - dx     ) * 2.0f / 16.0f;
	col += tex2D(smp, uv - dx + dy) * 1.0f / 16.0f;
	col += tex2D(smp, uv      - dy) * 2.0f / 16.0f;
	col += tex2D(smp, uv          ) * 4.0f / 16.0f;
	col += tex2D(smp, uv      + dy) * 2.0f / 16.0f;
	col += tex2D(smp, uv + dx - dy) * 1.0f / 16.0f;
	col += tex2D(smp, uv + dx     ) * 2.0f / 16.0f;
	col += tex2D(smp, uv + dx + dy) * 1.0f / 16.0f;
	return col;
}

float2 get_luminance(float4 col)
{
	// PAL, SECAM, BT.601, BT.709.
	return col.r * 0.299 + col.g * 0.587 + col.b * 0.114;
}

// HSV の V と S を x と y にそれぞれ得る.
float2 get_v_s(float4 col)
{
	float v = max(max(col.r, col.g), col.b);
	float s = v - min(min(col.r, col.g), col.b);
	return float2(v, s);
}

float easeInSine(float x) {
	return 1.0f - cos((x * 3.141592f) / 2.0f);
}

float easeOutQuad(float x) {
	return 1.0f - (1.0f - x) * (1.0f - x);
}

float4 around_sample(float2 uv, float2 dx, float2 dy)
{
	float4 ret = 0.0f;

	ret += tex2D(smp, uv - dx - dy) * 2.0f;
	ret += tex2D(smp, uv - dx     ) * 5.0f;
	ret += tex2D(smp, uv - dx + dy) * 2.0f;
	ret += tex2D(smp, uv      - dy) * 5.0f;
	ret += tex2D(smp, uv          ) * 6.0f;
	ret += tex2D(smp, uv      + dy) * 5.0f;
	ret += tex2D(smp, uv + dx - dy) * 2.0f;
	ret += tex2D(smp, uv + dx     ) * 5.0f;
	ret += tex2D(smp, uv + dx + dy) * 2.0f;

	ret += tex2D(smp, uv - 2.0f * dx) * 2.0f;
	ret += tex2D(smp, uv + 2.0f * dx) * 2.0f;
	ret += tex2D(smp, uv - 2.0f * dy) * 2.0f;
	ret += tex2D(smp, uv + 2.0f * dy) * 2.0f;

	ret /= 42.0f; // normalize.
	return ret;
}

float gamma_curve(float x, float gamma)
{
	return pow(x, gamma);
}

float contrast(float x)
{
	// cos とガンマカーブでコントラストを調整.
	// もっとマシな実装はある気がする.
	float ret;
	ret = -(cos(3.141592f * x) - 1.0f) / 2.0f;
	return gamma_curve(ret, 1.05f);
}

float4 ps_some_crt_effect(VS_Output input) : COLOR0
{
	float2 uv = input.tex;

	// 出力解像度でのピクセル位置.
	float ox = uv.x * outsz.x;
	float oy = uv.y * outsz.y;

	// RGB どの列かを求める.
	float ox_mod = frac(ox / 3.0f);
	float r_coeff = (ox_mod < 0.3) ? 1.0f : 0.0f;
	float g_coeff = (0.3 <= ox_mod && ox_mod < 0.6) ? 1.0f : 0.0f;
	float b_coeff = (0.6 <= ox_mod) ? 1.0f : 0.0f;

	// 3x3 の範囲の拡大画素で共通のサンプリング位置.
	float sx, sy;
	sx = (((ox / 3.0f) - frac(ox / 3.0f)) * 3.0f + 1.0f) / outsz.x;
	sy = (((oy / 3.0f) - frac(oy / 3.0f)) * 3.0f + 1.0f) / outsz.y;
	float2 s_uv = float2(sx, sy);
	float2 dx_s_v = float2(1.0f / tex0sz.x, 0);
	float2 dy_s_v = float2(0, 1.0f / tex0sz.y);
	// 3x3 範囲に共通した入力画素を取得.
	float4 icol = tex2D(smp, float2(sx, sy));
	float il = get_luminance(icol);

	// ガウシアンフィルタで使う隣のピクセルまでの距離.
	float d_g = 0.42f + easeInSine(icol) * 0.3f;
	float2 dx_g = float2(d_g / tex0sz.x, 0);
	float2 dy_g = float2(0, d_g / tex0sz.y);
	// ガウシアンフィルタを使ってサンプリング.
	float4 col = gaussian_sample(uv, dx_g, dy_g);

	// simple scanline.
	float my = frac(oy / 6.0f);
	float mx = frac(ox / 8.0f);
	if(my < 0.15f)
		col.rgb = col.rgb * 0.9f;
	else if(0.5 <= my && my < 0.6)
		col.rgb = col.rgb * 0.84f;
	else if(mx < 0.13f || 0.5f < mx && mx < 0.63f)
		col.rgb = col.rgb * 0.98f;
	// random noise.
	col.rgb = col.rgb * (1.0f - 0.05f * rand(float2(ox, oy)));

	// 出力 3x3 範囲の上端下端の画素を暗くする.
	// 横着してガンマカーブで実装.
	float oy_mod = frac(oy / 3.0f);
	if (oy < 0.4 || 0.6 <= oy)
	{
		col.r = gamma_curve(col.r, 1.14);
		col.g = gamma_curve(col.g, 1.14);
		col.b = gamma_curve(col.b, 1.14);
	}

	// 自分の輝度.
	float l0 = get_luminance(col);
	// 周囲の色.
	float4 acol = around_sample(s_uv, dx_s_v, dy_s_v);
	float l1 = get_luminance(acol);
	l1 = easeOutQuad(l1); // 少し持ち上げる.

	// 自分の明るさと周囲の明るさに応じてその列が担当しない RGB を増やす量.
	// 2倍の時は周囲の輝度も参照する(滲ませる).
	float4 adder = 0.55f * (1.5f * l0 + 0.25f * l1) / 1.75f;
	adder.r += (0.45f * acol.r);
	adder.g += (0.45f * acol.g);
	adder.b += (0.45f * acol.b);
	adder *= 0.81f;

	// 列毎に RGB 値のみ残す.
	float or = col.r * r_coeff;
	float og = col.g * g_coeff;
	float ob = col.b * b_coeff;
	// RGB 毎に増やす量を適用.
	or += g_coeff * adder.r + b_coeff * adder.r;
	og += b_coeff * adder.g + r_coeff * adder.g;
	ob += r_coeff * adder.b + g_coeff * adder.b;

	// 色味の調整.
	or *= 0.96f;
	og *= 0.93f;
	ob *= 1.0f;

	// コントラスト調整.
	or = contrast(or);
	og = contrast(og);
	ob = contrast(ob);

	return float4(or, og, ob, 1.0f);
}

technique TShader
{
	pass P0
	{
		VertexShader = compile vs_3_0 vs();
		PixelShader  = compile ps_3_0 ps_some_crt_effect();
	}
}

