texture tex0;

sampler smp0 = sampler_state
{
	Texture = <tex0>;
	MinFilter = LINEAR;
	MagFilter = LINEAR;
	MipFilter = NONE;
	AddressU = Clamp;
	AddressV = Clamp;
};

struct VS_Output
{
	float4 pos : POSITION;
	float2 tex : TEXCOORD0;
};

uniform float2 tex0sz;
uniform float2 outsz;

VS_Output vs(float4 pos : POSITION,
             float4 col : COLOR0,
             float2 tex : TEXCOORD0)
{
	VS_Output o;
	o.pos = pos;
	o.tex = tex;
	return o;
}

// orig の周囲 5x5 の間隔 d の格子の交点のうち何個が center にある半径 radius の円の中にあるか数える.
float count_around(float2 orig, float d, float2 center, float radius)
{
	float ret = 0.0f;
	for (int y = -2; y <= 2; ++y) {
		for (int x = -2; x <= 2; ++x) {
			const float p_x = orig.x + d * (float)x;
			const float p_y = orig.y + d * (float)y;

			const float distance = sqrt(pow(center.x - p_x, 2) + pow(center.y - p_y, 2));
			if (distance <= radius) {
				ret += 1.0f;
			}
		}
	}
	return ret;
}

float4 gaussian(sampler smp, float2 uv, float dx, float dy)
{
	const float2 dx2 = float2(dx, 0.0f);
	const float2 dy2 = float2(0.0f, dy);
	float4 ret = 0.0f;
	ret += tex2D(smp0, uv - dx2 - dy2) * 1.0f;
	ret += tex2D(smp0, uv       - dy2) * 2.0f;
	ret += tex2D(smp0, uv + dx2 - dy2) * 1.0f;
	ret += tex2D(smp0, uv - dx2      ) * 2.0f;
	ret += tex2D(smp0, uv            ) * 4.0f;
	ret += tex2D(smp0, uv + dx2      ) * 2.0f;
	ret += tex2D(smp0, uv - dx2 + dy2) * 1.0f;
	ret += tex2D(smp0, uv       + dy2) * 2.0f;
	ret += tex2D(smp0, uv + dx2 + dy2) * 1.0f;
	return ret / 16.0f;
}

float get_luminance(float4 col)
{
	// PAL, SECAM, BT.601, BT.709.
	return col.r * 0.299 + col.g * 0.587 + col.b * 0.114;
}

float2 get_sv(float4 col)
{
	// HSV の S と V を 戻り値の x, y にそれぞれ得る.
	float v = max(col.r, max(col.g, col.b));
	float2 ret;
	ret.x = v - min(col.r, min(col.g, col.b));
	ret.y = v;
	return ret;
}

float gamma_curve(float x, float gamma)
{
	return pow(x, gamma);
}

float contrast(float x)
{
	// cos とガンマカーブでコントラストを調整.
	// もっとマシな実装はある気がする.
	float ret;
	ret = -(cos(3.141592f * x) - 1.0f) / 2.0f;
	ret = gamma_curve(ret, 0.78f);
	return ret;
}

float4 ps_shadow_mask_crt_effect(VS_Output input) : COLOR0
{
	// ディスプレイのパラメータ.
	const float width_mm = 300.0f; // 想定ディスプレイ横幅.
	const float height_mm = width_mm * tex0sz.y / tex0sz.x;
	const float dot_pitch_mm = 0.43f; // 想定ドットピッチ.
	const float pitch_mm = (dot_pitch_mm / 2.0f) * (2.0f / sqrt(3.0f)); // RGB 関係なく隣り合う画素の距離.
	const float phi_mm = pitch_mm * 0.8f; // 想定シャドーマスク画素直径.
	const float line_dy_mm = pitch_mm * sqrt(3.0f) / 2.0f; // 画素の横のライン同士の距離.

	// レンダリング結果で1個のシャドーマスク画素を構成するピクセルの最小個数.
	const float n_min_member = 4.0f;

	// 他のパラメータ.
	const float out_px_per_mm = outsz.x / width_mm; // 想定ディスプレイの 1mm の幅が出力テクスチャ何ピクセル分か.
	const float mm_per_out_px = width_mm / outsz.x; // 出力テクスチャ1ピクセルの幅が想定ディスプレイ上で何 mm か.

	// 出力テクスチャでのピクセル位置.
	const float2 uv = input.tex;
	const float oy = uv.y * outsz.y;
	const float ox = uv.x * outsz.x;

	// 想定ディスプレイ上での位置.
	const float shear_y = 0.0f; // シャドーマスク画素が並ぶ角度を変えたい場合非0にする.
	const float y_mm = (uv.y * height_mm + uv.x * shear_y);
	const float x_mm = uv.x * width_mm;

	// どのシャドーマスク画素(shadow mask pixel)に属しているか計算する.
	const float smpxl_y = trunc(y_mm / line_dy_mm); // 何ライン目か.
	const bool even_line = frac(smpxl_y / 2.0f) < 0.5f; // ラインの偶奇判定用.
	const float smpxl_x = trunc((x_mm + (even_line ? 0.0f : (0.5f * pitch_mm))) / pitch_mm); // 奇数ラインで半ピッチ左にずれるものとする.

	// シャドーマスク画素中心の位置を求める.
	const float smcenter_y_mm = (smpxl_y + 0.5f) * line_dy_mm;
	const float smcenter_x_mm = (smpxl_x + (even_line ? 0.5f : 0.0f)) * pitch_mm;
	// シャドーマスク画素中心からの距離を求める.
	const float smdist_mm = sqrt(pow(smcenter_y_mm - y_mm, 2) + pow(smcenter_x_mm - x_mm, 2));

	// 画素の色.
	float r_coeff, g_coeff, b_coeff;
	// 周囲の画素が幾つ自分と同じシャドーマスク画素の中にあるか.
	float n_member;
	// シャドーマスク画素範囲外かどうか.
	bool out_of_smpxl = phi_mm / 2.0f < smdist_mm;
	if (out_of_smpxl) {
		r_coeff = g_coeff = b_coeff = 0.0f;
		n_member = 0.0f;
	}
	else {
		if (even_line) {
			const float fx = frac(smpxl_x / 3.0f); // 偶数ラインでは画面左端から RGBRGB... の順とする.
			r_coeff = (fx < 0.33f) ? 1.0f : 0.0f;
			g_coeff = (0.33f <= fx && fx < 0.66f) ? 1.0f : 0.0f;
			b_coeff = (0.66f <= fx) ? 1.0f : 0.0f;
		}
		else {
			const float fx = frac(smpxl_x / 3.0f); // 奇数ラインでは画面左端(に半分はみ出た画素)から GBRGBR... の順とする.
			g_coeff = (fx < 0.33f) ? 1.0f : 0.0f;
			b_coeff = (0.33f <= fx && fx < 0.66f) ? 1.0f : 0.0f;
			r_coeff = (0.66f <= fx) ? 1.0f : 0.0f;
		}

		// 明るさ調整.
		n_member = count_around(
			float2(x_mm, y_mm), mm_per_out_px,
			float2(smcenter_x_mm, smcenter_y_mm), phi_mm / 2.0f);
		if (n_member == n_min_member) {
			r_coeff *= 0.96f;
			g_coeff *= 0.93f;
			b_coeff *= 1.0f;
		}
		else {
			// 1個のシャドーマスク画素を構成する LCD 画素の組の明るさの合計が見た目上揃うようにする.
			r_coeff *= (n_min_member / n_member) * 1.2f * 0.96f;
			g_coeff *= (n_min_member / n_member) * 1.17f * 0.93f;
			b_coeff *= (n_min_member / n_member) * 1.25f * 1.0f;
		}
	}

	// 入力のサンプリング.
	const float4 icol = tex2D(smp0, uv);
	const float il = get_luminance(icol);
	const float2 isv = get_sv(icol);

	// 周囲のサンプリング.
	const float4 gcol = gaussian(smp0, uv, 3.6f / outsz.x, 3.6f / outsz.y);
	const float gl = get_luminance(gcol);
	const float2 gsv = get_sv(gcol);

	// 自分と周囲のの明るさと彩度に応じて画素が担当しない RGB を増やす量.
	const float4 extra_coeff =
		0.24f * il * (1.0f - isv.x) +
		0.12f * gl * (1.0f - gsv.x);

	// 周囲の明るさから求めた画素の値の最低値.
	const float min_rgb = (0.36f * gl * (1.0f - gsv.x)) * (out_of_smpxl ? 1.0f : 0.33f);

	// RGB 値のみ残す.
	float or = icol.r * r_coeff;
	float og = icol.g * g_coeff;
	float ob = icol.b * b_coeff;
	// 担当しない RGB の増やす量を適用.
	or += (g_coeff * extra_coeff.r + b_coeff * extra_coeff.r) * icol.r;
	og += (b_coeff * extra_coeff.g + r_coeff * extra_coeff.g) * icol.g;
	ob += (r_coeff * extra_coeff.b + g_coeff * extra_coeff.b) * icol.b;
	// 最低値の適用.
	or = max(or, min_rgb);
	og = max(og, min_rgb);
	ob = max(ob, min_rgb);

	// コントラスト調整.
	or = contrast(or);
	og = contrast(og);
	ob = contrast(ob);

	return float4(or, og, ob, 1.0f);
}

VertexShader vsvs = compile vs_3_0 vs();

technique TShader
{
	pass P0
	{
		VertexShader = vsvs;
		PixelShader = compile ps_3_0 ps_shadow_mask_crt_effect();
	}
}
