﻿open System.IO.Ports // NuGet からインストールする必要あり.
open System.Text.Json
open System.IO


// シリアルポート通信速度.
let rate = 9600

// portname で指定された COM ポートを返す.
// エラーが起きた時はエラー内容を表示して None を返す.
let getComPort portname rate =
    try
        use port = new SerialPort(portname, rate)
        port
    with ex ->
        printfn "Failed to get COM port (%s)" ex.Message
        reraise ()


// sent_str を port に書いて, その応答(CR が来るまで)を得る.
let sendAndReceive (port: SerialPort) (sent_str: string) =
    port.Write(sent_str)

    let mutable received = ""
    let mutable finished = false

    while not finished do
        let data = port.ReadExisting()

        if data.Length <> 0 then
            if data.Contains "\r" then
                finished <- true

            received <- received + data

    received


// コマンド生成時のエラー.
exception CommandGenearationError of string
// コマンドを json にする時のエラー.
exception CommandToJsonPartError of string


// タリーで返る受信ステータスに対応する列挙型.
type RecvStatus =
    | Ok = 'A'
    | Unknown = '1'
    | Error = '5'

// 入力のアスペクト比.
type InAspectRatio =
    | Auto = 0
    | FullAspect = 1
    | Aspect4_3 = 2
    | Aspect5_4 = 3
    | Aspect15_9 = 4
    | Aspect16_9 = 5
    | Aspect16_10 = 6
    | Aspect17_9 = 7

// オートセットアップ自動起動設定.
type InAutoSetupMode =
    | FullAuto = 0
    | Manual = 1
    | Off = 2

// 出力解像度.
type OutResolution =
    | D2 = -1 // D2.480p
    | Reso640x480 = 0 // 640x480
    | D4 = 3 // D4.720p
    | Reso1280x960 = 7 // 1280x960
    | D5 = 12 // D5.1080p
    | D3 = 14 // D3.1080i


// コマンドの設定値の json ファイルでの表現.
type CommandSaveType =
    | String
    | Number

// コマンドを表すレコード.
type Command =
    { cmd_1st: char
      cmd_2nd: char
      value: int32
      min: int32
      max: int32
      cmd_name: string
      save_type: CommandSaveType
      cmd_str: string }


// json に書かれている設定情報を表すレコード.
// 設定名と設定値の一覧.
type Setting =
    { Name: string; Commands: Command list }


// サポートしているコマンド一覧.
let supportedCommands: Command list = [
    // 文字列保存.
    { cmd_1st = 'A'; cmd_2nd = 'a'; value = 0; min = -167; max = 167; cmd_name = "in_aspect_ratio"; save_type = CommandSaveType.String; cmd_str = "" }
    { cmd_1st = 'D'; cmd_2nd = 'i'; value = 0; min = 0; max = 2; cmd_name = "in_auto_setup_mode"; save_type = CommandSaveType.String; cmd_str = "" }
    { cmd_1st = 'E'; cmd_2nd = 'a'; value = 0; min = -1; max = 20; cmd_name = "out_resolusion"; save_type = CommandSaveType.String; cmd_str = "" }
    // 数値保存.
    { cmd_1st = 'A'; cmd_2nd = 'c'; value = 0; min = 0; max = 63; cmd_name = "in_clock_phase"; save_type = CommandSaveType.Number; cmd_str = "" }
    { cmd_1st = 'A'; cmd_2nd = 'd'; value = 0; min = -500; max = 500; cmd_name = "in_total_h_clock"; save_type = CommandSaveType.Number; cmd_str = "" }
    { cmd_1st = 'A'; cmd_2nd = 'e'; value = 0; min = -800; max = 800; cmd_name = "in_shift_h_pixel"; save_type = CommandSaveType.Number; cmd_str = "" }
    { cmd_1st = 'A'; cmd_2nd = 'f'; value = 0; min = -500; max = 500; cmd_name = "in_shift_v_pixel"; save_type = CommandSaveType.Number; cmd_str = "" }
    { cmd_1st = 'A'; cmd_2nd = 'g'; value = 0; min = -401; max = 401; cmd_name = "in_reso_h_pixel"; save_type = CommandSaveType.Number; cmd_str = "" }
    { cmd_1st = 'A'; cmd_2nd = 'h'; value = 0; min = -501; max = 501; cmd_name = "in_reso_v_pixel"; save_type = CommandSaveType.Number; cmd_str = "" }
    { cmd_1st = 'B'; cmd_2nd = 'a'; value = 0; min = -1000; max = 7000; cmd_name = "in_zoom_size"; save_type = CommandSaveType.Number; cmd_str = "" }
    { cmd_1st = 'D'; cmd_2nd = 'e'; value = 0; min = -30; max = 30; cmd_name = "in_adc_r"; save_type = CommandSaveType.Number; cmd_str = "" }
    { cmd_1st = 'D'; cmd_2nd = 'f'; value = 0; min = -30; max = 30; cmd_name = "in_adc_g"; save_type = CommandSaveType.Number; cmd_str = "" }
    { cmd_1st = 'D'; cmd_2nd = 'g'; value = 0; min = -30; max = 30; cmd_name = "in_adc_b"; save_type = CommandSaveType.Number; cmd_str = "" }
    { cmd_1st = 'E'; cmd_2nd = 'f'; value = 0; min = -50; max = 50; cmd_name = "back_lumi"; save_type = CommandSaveType.Number; cmd_str = "" }
    { cmd_1st = 'E'; cmd_2nd = 'g'; value = 0; min = 0; max = 105; cmd_name = "back_color"; save_type = CommandSaveType.Number; cmd_str = "" }
    { cmd_1st = 'E'; cmd_2nd = 'h'; value = 0; min = 0; max = 359; cmd_name = "back_hue"; save_type = CommandSaveType.Number; cmd_str = "" }
]


// cmd_str を json の一部になる形式の文字列に変換する.
// supportedCommands に cmd.cmd_name が含まれているかチェックしていないので注意.
let cmdStrToJsonPart (cmd: Command) =
    let v = System.Int32.Parse(cmd.cmd_str[3 .. 8])

    let mutable ret: string = "\"" + cmd.cmd_name + "\": "
    if cmd.save_type = CommandSaveType.Number then
        ret <- ret + v.ToString()
    else
        match cmd.cmd_name with
        | "in_aspect_ratio" -> ret <- ret + "\"" + enum<InAspectRatio>(v).ToString() + "\""
        | "in_auto_setup_mode" -> ret <- ret + "\"" + enum<InAutoSetupMode>(v).ToString() + "\""
        | "out_resolusion" -> ret <- ret + "\"" + enum<OutResolution>(v).ToString() + "\""
        | _ -> raise (CommandToJsonPartError(sprintf "Unknown command name (%s)." cmd.cmd_name))
    ret


// 文字列をその文字列で表される列挙型のメンバに変換する.
let getEnum<'T> (value: string) =
    System.Enum.Parse(typedefof<'T>, value, true) :?> 'T


// name の名前で値に value が入った Command を返す(value は文字列).
let setCommandValueString (name: string) (value: string) =
    let cmd = List.find (fun c -> c.save_type = CommandSaveType.String && c.cmd_name = name) supportedCommands
    try
        match name with
        | "in_aspect_ratio" -> { cmd with value = (unbox<int32> (getEnum<InAspectRatio> value)) }
        | "out_resolusion" -> { cmd with value = (unbox<int32> (getEnum<OutResolution>(value))) }
        | "in_auto_setup_mode" -> { cmd with value = (unbox<int32> (getEnum<InAutoSetupMode> value)) }
        | _ -> raise (CommandGenearationError(sprintf "Unknown property name (%s)." name))
    with ex ->
        raise (CommandGenearationError(sprintf "Failed to generate command of \"%s\" (%s)." name ex.Message))


// name の名前で値に value が入った Command を返す(value は数値).
let setCommandValueInt32 (name: string) (value: int32) =
    let cmd = List.find (fun c -> c.save_type = CommandSaveType.Number && c.cmd_name = name) supportedCommands
    if value < cmd.min then
        raise (CommandGenearationError(sprintf "Value is smaller than minimal value (%s)." name))
    if cmd.max < value then
        raise (CommandGenearationError(sprintf "Value is larger than maximum value (%s)." name))
    { cmd with value = value }


// コマンドとして送る文字列を作る.
// cmd.value に値を入れておくと cmd_str に文字列が入る.
let makeCommandStr (cmd: Command) (get: bool) =
    let c1: char = if get then System.Char.ToLower cmd.cmd_1st else System.Char.ToUpper cmd.cmd_1st
    let c2: char = if get then System.Char.ToUpper cmd.cmd_2nd else System.Char.ToLower cmd.cmd_2nd

    let mutable s: string = "#" + c1.ToString() + c2.ToString()
    if cmd.value = 0 then
        s <- s + "00000\r"
    else
        s <- s + (sprintf "%+05d\r" cmd.value)

    { cmd with cmd_1st = c1; cmd_2nd = c2; cmd_str = s }


// path で指定された json ファイルの中身を複数の一連のコマンドに変換して返す.
let loadJsonFile path =
    // JsonElements を Setting にする.
    let convertToSetting (elem: JsonElement) =
        let mutable cmds = new ResizeArray<Command>()

        // cmds にコマンドを入れる.
        for o in elem.EnumerateObject() do
            let name = o.Name

            if name <> "name" && name <> "comment" then
                let value_kind = o.Value.ValueKind

                match value_kind with
                | JsonValueKind.String ->
                    let value = o.Value.GetString()
                    cmds.Add <| makeCommandStr (setCommandValueString name value) false
                | JsonValueKind.Number ->
                    let value = o.Value.GetInt32()
                    cmds.Add <| makeCommandStr (setCommandValueInt32 name value) false
                | _ -> ()

        { Name = elem.GetProperty("name").GetString()
          Commands = cmds.ToArray() |> Array.toList }

    try
        use fs = new FileStream(path, FileMode.Open, FileAccess.Read)
        use sr = new StreamReader(fs)

        let json_txt = sr.ReadToEnd()
        let json: JsonElement = JsonSerializer.Deserialize json_txt

        let mutable settings: Setting list =
            match json.TryGetProperty("settings") with
            | (true, contents) ->
                let seq = contents.EnumerateArray()
                Seq.map convertToSetting seq |> Seq.toList
            | (false, _) -> []

        settings
    with ex ->
        printfn "Failed to load json file (%s)" ex.Message
        reraise ()


// set コマンドの処理.
let execSetCommand port_name json_path setting_name =
    try
        let all_settings = loadJsonFile json_path

        let settings =
            List.choose
                (fun elem ->
                    match elem with
                    | elem when elem.Name = setting_name -> Some(elem)
                    | _ -> None)
                all_settings

        let n_matched = settings.Length

        if n_matched = 0 then
            printfn "setting \"%s\" not found" setting_name
            printfn "valid names:"

            for s in all_settings do
                printfn "  %s" s.Name

            1
        elif 2 <= n_matched then
            printfn "setting \"%s\" was defiend multiple times" setting_name
            1
        else
            let com_port = getComPort port_name rate
            com_port.Open()

            for c in settings.[0].Commands do
                printfn "Send: %s -> %s" c.cmd_str[.. c.cmd_str.Length - 2]
                <| sendAndReceive com_port c.cmd_str

            com_port.Close()
            0
    with ex ->
        printfn "set command failed (%s)" ex.Message
        1


// get コマンドの処理.
let execGetCommand port_name =
    try
        let com_port = getComPort port_name rate
        com_port.Open()

        let mutable results = new ResizeArray<Command>()

        // サポートしているコマンドの値を全部読み出す.
        for cmd in supportedCommands do
            let c = makeCommandStr {cmd with value = 0} true
            let r = sendAndReceive com_port c.cmd_str
            results.Add({ c with cmd_str = r })
            printfn "%s -> %s" c.cmd_str[.. c.cmd_str.Length - 2] r

        // 読んだ結果を json に変換して表示する.
        printfn "--- json ---\n{\n\t\"name\": \"\"\n\t\"comment\": \"\""
        for i in 0 .. results.Count - 1 do
            if (i + 1) = results.Count then
                printfn "\t%s" <| cmdStrToJsonPart results.[i]
            else
                printfn "\t%s," <| cmdStrToJsonPart results.[i]
        printfn "}"

        com_port.Close()
        0
    with ex ->
        printfn "get command failed (%s)" ex.Message
        1


[<EntryPoint>]
let main args =
    let argc = args.Length

    if argc = 0 then
        printfn "RS-1500A Controller ver 1.0a 2024/08/24"
        printfn "USAGE: r15actrl [command] (command args...)"
        printfn "  available command: {set, get}"
        1
    else
        let command = args.[0]

        if command = "set" then
            if argc <> 4 then
                printfn "USAGE: r15actrl set [com_port] [setting_json] [setting_name]"
                1
            else
                execSetCommand args.[1] args.[2] args.[3]
        elif command = "get" then
            if argc <> 2 then
                printfn "USAGE: r15actrl get [com_port]"
                1
            else
                execGetCommand args.[1]
        else
            printfn "Unknown command"
            1
